<?php


namespace App\Service;


use App\Entity\Attendance;
use App\Entity\Classes;
use App\Entity\ClassSessionEnrolment;
use App\Entity\ClassSessionEnrolmentSubjects;
use App\Entity\ClassSubject;
use App\Entity\EmailTemplateEvents;
use App\Entity\EmailTemplates;
use App\Entity\ExamsClassesStudent;
use App\Entity\Guardian;
use App\Entity\Invoices;
use App\Entity\ShortEmailTemplates;
use App\Entity\User;
use App\Repository\EmailTemplateEventsRepository;
use App\Repository\EmailTemplatesRepository;
use App\Service\EmailSettings\Base;
use App\Service\SMSGateways\OperationalionalGatewayClass;
use Doctrine\Common\Persistence\ObjectManager;
use Exception as ExceptionAlias;
use Gedmo\Translatable\Document\Repository\TranslationRepository;
use Gedmo\Translatable\Entity\Translation;
use Gedmo\Translatable\Translatable;
use Symfony\Bundle\FrameworkBundle\Templating\EngineInterface;
use Symfony\Bundle\TwigBundle\TwigBundle;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\PropertyAccess\PropertyAccess;
use Symfony\Component\PropertyAccess\PropertyAccessor;
use Twig\Template;

class TriggerNotifications
{

    /**
     * @var EmailTemplateEvents
     */
    private $email_template_events;
    /**
     * @var EmailTemplates
     */
    private $email_templates;
    /**
     * @var EmailTemplateEventsRepository
     */
    private $email_template_events_repository;
    /**
     * @var EmailTemplatesRepository
     */
    private $email_templates_repository;
    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var AppSettings
     */
    private $app_settings;
    /**
     * @var Constants
     */
    private $constants;
    /**
     * @var ObjectManager
     */
    private $object_manager;
    /**
     * @var \Twig_Environment
     */
    private $twig_environment;
    /**
     * @var OperationalionalGatewayClass
     */
    private $operationalionalGatewayClass;
    /**
     * @var Base
     */
    private $EmailBase;

    /**
     * TriggerNotifications constructor.
     *
     * @param Base                          $EmailBase
     * @param OperationalionalGatewayClass  $operationalionalGatewayClass
     * @param EmailTemplateEventsRepository $email_template_events_repository
     * @param EmailTemplatesRepository      $email_templates_repository
     * @param DefaultFunction               $default_function
     * @param AppSettings                   $app_settings
     * @param Constants                     $constants
     * @param ObjectManager                 $object_manager
     * @param \Twig_Environment             $twig_environment
     */
    public function __construct(
        Base $EmailBase,
        OperationalionalGatewayClass $operationalionalGatewayClass,
        EmailTemplateEventsRepository $email_template_events_repository,
        EmailTemplatesRepository $email_templates_repository,
        DefaultFunction $default_function,
        AppSettings $app_settings,
        Constants $constants,
        ObjectManager $object_manager,
        \Twig_Environment $twig_environment
    ) {
        $this->email_template_events_repository = $email_template_events_repository;
        $this->email_templates_repository       = $email_templates_repository;
        $this->default_function                 = $default_function;
        $this->app_settings                     = $app_settings;
        $this->constants                        = $constants;
        $this->object_manager                   = $object_manager;
        $this->twig_environment                 = $twig_environment;
        $this->operationalionalGatewayClass     = $operationalionalGatewayClass;
        $this->EmailBase                        = $EmailBase;
    }

    ## method is used to send email
    ## notification will be trigger if template is active.


    /**
     * @param $args
     * === $argument list ===
     *
     *  Name          | type of variable | variable name
     * --------------------------------------------
     * event name | string | event_name
     * user object who will receive the email | App\Entity|User , App\Entity\Guardian | user
     *triggerNotifications
     * Phone: to send the messages
     *
     * @return EmailTemplateEvents|array|mixed
     */
    public function triggerNotifications($args)
    {

        $response = [];

        ## if event name is missing then return with the error.
        if (empty($args['event_name'])) {
            return $response = $this->default_function->push_error($response, 'Event name is empty');
        }


        ## get the template reference on base of event.
        try {
            /** @var EmailTemplateEvents $templateEvent */
            $templateEvent = $this->email_template_events_repository->findOneBy(['event_name' => $args['event_name']]);
        } catch (ExceptionAlias $exception) {
            ## push error if we've
            return $this->default_function->push_error($response, $exception->getMessage());
        }

        ## if event not found in the database.
        if (empty($templateEvent)) {
            return $this->default_function->push_error(
                $response,
                $args['event_name'].' event is not found in the database'
            );
        }

        ## if we found the event in the database then get the attached / related template.
        /** @var  EmailTemplates $template [0] */
        $template = $templateEvent->getTriggerTemplate()->toArray();

        ## validate, if we found the template
        if (empty($template[0])) {
            return $this->default_function->push_error(
                $response,
                'No, template found against '.$args['event_name'].' event'
            );
        }

        ## validate, template is active or in-active.
        if ( ! $template[0]->getIsActive()) {
            return $this->default_function->push_error(
                $response,
                'To send notifications, please activate template '.$template[0]->getName()
            );
        }

        ## if we found the template against the event then get the translated template, but before that get the application default language.
        $appDefaultLang = $this->app_settings->getAppSettingsFromSessions(
            $this->constants::AppSettingsLabels['default_language']
        );

        ## get the translated template from the database.
        /** @var TranslationRepository $translationRepo */
        $translationRepo = $this->object_manager->getRepository('Gedmo\\Translatable\\Entity\\Translation');
        ## fetching translation of given, template.
        $transitionedTemplates = $translationRepo->findTranslations($template[0]);
        ## sending email

        $this->preSendEmail($transitionedTemplates, $appDefaultLang, $template, $args);

        ## sending sms
        ## getting SMS template from database.
        $shortEmailTemplate = $template[0]->getShortEmailTemplates();
        if ($shortEmailTemplate instanceof ShortEmailTemplates && $shortEmailTemplate->getIsActive(
            ) && ! empty($args['phone'])) {
            ## finding translation of sms
            $transitionedTemplates = $translationRepo->findTranslations($shortEmailTemplate);
            $translatedTemplates   = $this->getTemplateContentByCalculateTranslations(
                $transitionedTemplates,
                $appDefaultLang,
                $shortEmailTemplate
            );
            $bag                   = new ParameterBag();

            $content = null;


            if ( ! empty($args['notification_type'])) {
                switch ($args['notification_type']) {
                    case 'daily_attendance':
                        $content = $this->contentForAttendance(
                            $args['attendance'],
                            $args['user'],
                            $translatedTemplates['content'],
                            $args
                        );
                        break;
                    case 'when_invoice_creates':
                        $content = $this->contentForInvoiceGenerate(
                            $args['invoice'],
                            $args['user'],
                            $translatedTemplates['content'],
                            $args
                        );
                        break;
                    case 'exam_marks':
                        $content = $this->contentForMarksEnterForExam(
                            $args['examClassStudent'],
                            $args['user'],
                            $translatedTemplates['content'],
                            $args
                        );
                        break;
                    case 'class_schedule_type':
                        $content = $this->contentForClassSchedule(
                            $args['user'],
                            $translatedTemplates['content'],
                            $args
                        );
                        break;
                    default:
                        $content = $translatedTemplates['content'];
                }
            }

            $bag->set(
                'message',
                $content
            );
            $bag->set('to', $args['phone']);

            $this->sendMessage($bag);
        }

        ## return the email response.
        return $response;
    }


    /**
     * PURPOSE: Before sending email to user, system will check which translation is enabled & has templates is
     * translated in that system or not.. if not it will get the raw template and sent to the user.
     *
     * @param $transitionedTemplates
     * @param $appDefaultLang
     * @param $template
     *
     * @return mixed
     */
    public function getTemplateContentByCalculateTranslations($transitionedTemplates, $appDefaultLang, $template)
    {
        ## validate, template is translated in default language or not
        if (empty($transitionedTemplates[$appDefaultLang])) {
            ## if default language translation is not found then check for the english language.
            if (empty($transitionedTemplates['en'])) {
                ## if translation is not found in the english language then get the fekara written template.
                $finalTemplatesContent = $template;
            } else {
                ## if we found the template in the english version
                $finalTemplatesContent = $transitionedTemplates['en'];
            }
        } else {
            $finalTemplatesContent = $transitionedTemplates[$appDefaultLang];
        }

        return $finalTemplatesContent;
    }


    /**
     * ======================= SMS SENDING
     * =======================
     */

    /**
     * @param ParameterBag $parameterBag
     *
     * @return string
     */
    public function sendMessage(ParameterBag $parameterBag)
    {
        return $this->operationalionalGatewayClass->sendMessage($parameterBag);
    }


    /**
     * ======================= EMAIL SENDING
     * =======================
     * */

    /**
     *  Purpose: before sending email system will get the subject and content of the body.
     *
     * @param $transitionedTemplates
     * @param $appDefaultLang
     * @param $template
     * @param $args
     *
     * @return array|bool|mixed|string
     */
    public function preSendEmail($transitionedTemplates, $appDefaultLang, $template, $args)
    {
        $finalTemplatesContent = $this->getTemplateContentByCalculateTranslations(
            $transitionedTemplates,
            $appDefaultLang,
            $template[0]
        );

        ## if $finalTemplatesContent variable is array or object
        if (gettype($finalTemplatesContent) === 'array') {
            $content = $finalTemplatesContent['content'];
            $subject = $finalTemplatesContent['subject'];
        } else {
            /** @var EmailTemplates $finalTemplatesContent */
            $content = $finalTemplatesContent->getSubject();
            $subject = $finalTemplatesContent->getContent();
        }

        ## sending email
        $sendEmailArgs = [
            ## if we found current user email address then send email to user else send email to sulaman.
            'senderAddress'  => empty($args['receiver']) ? (! empty(
            $args['user']->getEmail()
            ) ? $args['user']->getEmail() : 'symikhan70@gmail.com') : $args['receiver'],
            ## subject of the template
            'subject'        => $subject,
            ## content of the template
            'content'        => $content,
            ## current user which will be receiver of this email
            'user'           => ! empty($args['user']) ? $args['user'] : '',
            ## additional info
            'additionalInfo' => ! empty($args['additionalInfo']) ? $args['additionalInfo'] : [],
            ## attachments with emails
            'attachments'    => empty($args['attachments']) ? null : $args['attachments'],
        ];

        $sendEmailArgs = array_merge($sendEmailArgs, $args);

        return $this->sendEmail($sendEmailArgs);
    }

    public function contentForAttendance(Attendance $attendanceDetails, User $userInfo, $content, $extraArr)
    {
        /** @var Guardian $guardian */
        $guardian = $extraArr['guardian'];

        $ar = [
            '{{ s_fname }}'             => $userInfo->getFname(),
            '{{ s_lname }}'             => $userInfo->getLName(),
            '{{ guardianName }}'        => $guardian->getFullName(),
            '{{ s_attendance_date }}'   => date_format($attendanceDetails->getDatetimeIn(), 'd-M-Y H:i'),
            '{{ s_attendance_status }}' => $extraArr['attendance_status_name'],
        ];

        if ( ! empty($attendanceDetails->getClassSessionEnrolment()->getClassSession()->getName())) {
            $ar['{{ s_current_class }}'] = $attendanceDetails->getClassSessionEnrolment()->getClassSession()->getName();
        }

        if ($attendanceDetails->getClassSessionEnrolmentSubject(
            ) instanceof ClassSessionEnrolmentSubjects && $attendanceDetails->getClassSessionEnrolmentSubject()
                                                                            ->getClassSubject(
                                                                            ) instanceof ClassSubject) {
            $ar['{{ s_subject }}'] = $attendanceDetails->getClassSessionEnrolmentSubject()
                                                       ->getClassSubject()
                                                       ->getSubjectName();
        } else {
            $ar['{{ s_subject }}'] = '';
        }


        $content = strtr($content, $ar);

        return $content;

    }

    public function contentForSingleStudentClassAttendance(
        User $userInfo,
        $content,
        $extraArr
    ) {
        /** @var Guardian $guardian */
        $guardian = $extraArr['guardian'];

        $ar = [
            '{{ s_fname }}'          => $userInfo->getFname(),
            '{{ s_lname }}'          => $userInfo->getLName(),
            '{{ guardianName }}'     => $guardian->getFullName(),
            '{{ attendance_table }}' => $extraArr['s_attendance_class_table'],
        ];

        $content = strtr($content, $ar);

        return $content;

    }

    public function contentForSingleStudentClassExam(
        User $userInfo,
        $content,
        $extraArr
    ) {
        /** @var Guardian $guardian */
        $guardian = $extraArr['guardian'];

        $ar = [
            '{{ s_fname }}'             => $userInfo->getFname(),
            '{{ s_lname }}'             => $userInfo->getLName(),
            '{{ guardianName }}'        => $guardian->getFullName(),
            '{{ s_marks_class_table }}' => $extraArr['s_marks_class_table'],
        ];

        $content = strtr($content, $ar);

        return $content;

    }


    public function contentForClassSchedule(
        User $userInfo,
        $content,
        $extraArr
    ) {
        /** @var Guardian $guardian */
        $guardian = $extraArr['guardian'];

        $ar = [
            '{{ s_fname }}'             => $userInfo->getFname(),
            '{{ s_lname }}'             => $userInfo->getLName(),
            '{{ guardianName }}'        => $guardian->getFullName(),
            '{{ s_current_class }}'     => $extraArr['class_name'],
            '{{ schedule_html_table }}' => $extraArr['schedule_html_table'],
        ];

        $content = strtr($content, $ar);

        return $content;

    }

    public function contentForInvoiceGenerate(Invoices $invoiceDetails, User $userInfo, $content, $extraArr)
    {
        /** @var Guardian $guardian */
        $guardian = $extraArr['guardian'];

        $ar = [
            '{{ s_fname }}'            => $userInfo->getFname(),
            '{{ s_lname }}'            => $userInfo->getLName(),
            '{{ guardianName }}'       => $guardian->getFullName(),
            '{{ s_invoice_due_date }}' => date_format($invoiceDetails->getInvDueDate(), 'd-M-Y H:i'),
            '{{ s_invoice_no }}'       => $invoiceDetails->getInvReference(),
            '{{ s_invoice_status }}'   => $invoiceDetails->getTheInvoiceStatus(),
            '{{ s_current_class }}'    => $invoiceDetails->getTheInvoiceStatus(),
            '{{ s_subject }}'          => $invoiceDetails->getTheInvoiceStatus(),
            '{{ s_invoice_amount }}'   => $invoiceDetails->getTheInvoiceAmountWhichIsNotPaid(),
        ];

        if ($invoiceDetails->getClassSessionEnrolment(
            ) instanceof ClassSessionEnrolment && $invoiceDetails->getClassSessionEnrolment()->getClassSession(
            ) instanceof Classes) {
            $ar['{{ class_session_enrolment_id }}'] = $invoiceDetails->getClassSessionEnrolment()
                                                                     ->getClassSession()
                                                                     ->getName();
        } else {
            $ar['{{ class_session_enrolment_id }}'] = '';
        }


        $content = strtr($content, $ar);

        return $content;

    }

    public function contentForMarksEnterForExam(
        ExamsClassesStudent $examsClassesStudent,
        User $userInfo,
        $content,
        $extraArr
    ) {
        /** @var Guardian $guardian */
        $guardian = $extraArr['guardian'];

        $ar = [
            '{{ s_fname }}'            => $userInfo->getFname(),
            '{{ s_lname }}'            => $userInfo->getLName(),
            '{{ guardianName }}'       => $guardian->getFullName(),
            '{{ s_current_class }}'    => $extraArr['class_name'],
            '{{ s_subject }}'          => $extraArr['subject_name'],
            '{{ s_obtain_marks }}'     => $examsClassesStudent->getEcsGainMarks(),
            '{{ s_total_marks }}'      => $extraArr['total_marks'],
            '{{ s_examDate }}'         => $extraArr['exam_at'],
            '{{ s_attendance_date }}'  => $extraArr['exam_at'],
            '{{ exam_attend_status }}' => $extraArr['exam_attend_status'],
        ];

        $content = strtr($content, $ar);

        return $content;

    }


    /**
     * Purpose: sending email to the user, This function parse variables in the templates as well.
     *
     * @param $args
     *
     * @return array|bool|mixed|string
     */
    public function sendEmail($args)
    {
        $bag = new ParameterBag();

        if ( ! empty($args['notification_type'])) {
            switch ($args['notification_type']) {
                case 'daily_attendance':
                    $args['content'] = $this->contentForAttendance(
                        $args['attendance'],
                        $args['user'],
                        $args['content'],
                        $args
                    );
                    break;
                case 'when_invoice_creates':
                    $args['content'] = $this->contentForInvoiceGenerate(
                        $args['invoice'],
                        $args['user'],
                        $args['content'],
                        $args
                    );
                    break;
                case 'exam_marks':
                    $args['content'] = $this->contentForMarksEnterForExam(
                        $args['examClassStudent'],
                        $args['user'],
                        $args['content'],
                        $args
                    );
                    break;
                case 'bulk_attendance_report':
                    $args['content'] = $this->contentForSingleStudentClassAttendance(
                        $args['user'],
                        $args['content'],
                        $args
                    );
                    break;
                case 'bulk_exam_report':
                    $args['content'] = $this->contentForSingleStudentClassExam(
                        $args['user'],
                        $args['content'],
                        $args
                    );
                    break;
                case 'class_schedule_type':
                    $args['content'] = $this->contentForClassSchedule(
                        $args['user'],
                        $args['content'],
                        $args
                    );
                    break;
            }
        }


        $bag->set('receiver', $args['senderAddress']);
        $bag->set('message', $args['content']);
        $bag->set('subject', trim(strip_tags($args['subject'])));
        $bag->set('user', $args['user']);

        return $this->EmailBase->sendEmail($bag);

        /**
         * @Deprecated 2019-11-24
         * */ //		$response = [];
        //		## if send is empty
        //		if ( empty( $args['senderAddress'] ) ) {ß
        //			return $response = $this->default_function->push_error( $response, 'Sender is empty' );
        //		}
        //		## sending email from
        //		$from = "fekara@sms.com";
        //		## set the sender address.
        //		$to = $args['senderAddress'];
        //		## find variables in the email content & parse them into relevant data.
        //		preg_match_all( "#\{{(.*?)\}}#", $args['content'], $variableInTemplate );
        //		/** @var User $user */
        //		$user = $args['user'];
        //		if ( ! empty( $user ) ) {
        //			$ar = [
        //				'{{ fname }}' => $user->getFname(),
        //				'{{ lname }}' => $user->getLName(),
        //			];
        //
        //			$args['content'] = strtr( $args['content'], $ar );
        //		}
        //		## replace additional info.
        //		if ( ! empty( $args['additionalInfo'] ) ) {
        //			$args['content'] = strtr( $args['content'], $args['additionalInfo'] );
        //		}
        //		## try to render the template.
        //		try {
        //			$response = $this->twig_environment->render( 'notification/emailNotifier.html.twig', [
        //				'content' => $args['content'],
        //			] );
        //		} catch ( ExceptionAlias $exception ) {
        //			return $response__ = $exception->getMessage();
        //		}
        //		$headers = "From: fekara@sms.com\r\n";
        //		$headers .= "Reply-To: no-reply@fekara.com\r\n";
        //		$headers .= "CC: sulaman@sulaman.pk\r\n";
        //		$headers .= "MIME-Version: 1.0\r\n";
        //		$headers .= "Content-Type: text/html; charset=UTF-8\r\n";
        //		try {
        //			$response = mail( $to, $args['subject'], $response, $headers );
        //		} catch ( ExceptionAlias $exception ) {
        //			$response = $exception->getMessage();
        //		}
        //		return $response;
    }


    public function logging($message, $phoneNumber)
    {
        $projectDir = realpath(dirname(__DIR__.'/../../../'));
        $logFile    = $projectDir.'/sendMessagesLogs.log';


        if (is_array($message)) {

            file_put_contents(
                $logFile,
                print_r($message, false),
                FILE_APPEND
            );
        } else {
            file_put_contents(
                $logFile,
                $message.' to '.$phoneNumber.' phone'.PHP_EOL,
                FILE_APPEND
            );
        }
    }

    public function loggingEmail($message, $email)
    {
        $projectDir = realpath(dirname(__DIR__.'/../../../'));
        $logFile    = $projectDir.'/sendMessagesLogs.log';


        if (is_array($message)) {

            file_put_contents(
                $logFile,
                print_r($message, false),
                FILE_APPEND
            );
        } else {
            file_put_contents(
                $logFile,
                $message.' to '.$email.' email'.PHP_EOL,
                FILE_APPEND
            );
        }
    }


}
