<?php

/*
 * All files related to the app settings will be upload in the public folder
 *
 * */

namespace App\Service;


use App\Entity\ClassTemplate;
use App\Entity\Guardian;
use App\Entity\Session;
use App\Entity\Subjects;
use App\Entity\User;
use App\Repository\AccountHeadsRepository;
use App\Repository\AccountsRepository;
use App\Repository\AppSettingsRepository;
use App\Repository\AttendanceStatusRepository;
use App\Repository\ClassFeeScheduleRepository;
use App\Repository\ClassTemplateRepository;
use App\Repository\PaymentMethodsRepository;
use App\Repository\SessionRepository;
use App\Repository\StudentFeeRepository;
use App\Repository\SubjectsRepository;
use Doctrine\DBAL\Connection as ConnectionAlias;
use Doctrine\DBAL\DBALException as DBALExceptionAlias;
use Doctrine\ORM\EntityManagerInterface;
use Facebook\WebDriver\Exception\ExpectedException;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;

class AppSettings
{


    /**
     * @var AppSettingsRepository
     */
    private $app_settings_repository;
    /**
     * @var Constants
     */
    private $constants;
    /**
     * @var EntityManagerInterface
     */
    private $entity_manager;
    /**
     * @var FileManagment
     */
    private $file_managment;
    /**
     * @var RequestStack
     */
    private $request;
    /**
     * @var PaymentMethodsRepository
     */
    private $payment_methods_repository;
    /**
     * @var AccountHeadsRepository
     */
    private $account_heads_repository;
    /**
     * @var StudentFeeRepository
     */
    private $student_fee_repository;
    /**
     * @var SubjectsRepository
     */
    private $subjects_repository;
    /**
     * @var ClassTemplateRepository
     */
    private $class_template_repository;
    /**
     * @var SessionRepository
     */
    private $session_repository;
    /**
     * @var AttendanceStatusRepository
     */
    private $attendanceStatusRepository;
    /**
     * @var AccountsRepository
     */
    private $accountsRepository;
    /**
     * @var ClassFeeScheduleRepository
     */
    private $classFeeScheduleRepository;


    public function __construct(
        AppSettingsRepository $app_settings_repository,
        Constants $constants,
        EntityManagerInterface $entity_manager,
        FileManagment $file_managment,
        RequestStack $request,
        PaymentMethodsRepository $payment_methods_repository,
        AccountHeadsRepository $account_heads_repository,
        StudentFeeRepository $student_fee_repository,
        SubjectsRepository $subjects_repository,
        ClassTemplateRepository $class_template_repository,
        SessionRepository $session_repository,
        AttendanceStatusRepository $attendanceStatusRepository,
        AccountsRepository $accountsRepository,
        ClassFeeScheduleRepository $classFeeScheduleRepository
    ) {
        $this->app_settings_repository = $app_settings_repository;
        $this->constants = $constants;
        $this->entity_manager = $entity_manager;
        $this->file_managment = $file_managment;
        $this->request = $request;
        $this->payment_methods_repository = $payment_methods_repository;
        $this->account_heads_repository = $account_heads_repository;
        $this->student_fee_repository = $student_fee_repository;
        $this->subjects_repository = $subjects_repository;
        $this->class_template_repository = $class_template_repository;
        $this->session_repository = $session_repository;
        $this->attendanceStatusRepository = $attendanceStatusRepository;
        $this->accountsRepository = $accountsRepository;
        $this->classFeeScheduleRepository = $classFeeScheduleRepository;
    }

    /**
     *  ==================== BEGIN:: Save language settings.
     * Purpose: Suppose application provides 50 languages translation & user only wants to display only 10 languages
     * translation then he/she will save his/her preference languauges by using this method.
     *
     * @return string
     * @var ParameterBag $requestBag
     *
     */
    public function saveUserPreferLocal($requestBag)
    {
        ## prerequisite checks.
        ## if multilingual option is not enabled.
        if (!$requestBag->get($this->constants::AppSettingsLabels['is_multilingual'])) {
            $requestBag->set($this->constants::AppSettingsLabels['is_multilingual'], 'off');
        }

        ## if page layout left to right is not selected.
        if (!$requestBag->get($this->constants::AppSettingsLabels['rtl'])) {
            $requestBag->set($this->constants::AppSettingsLabels['rtl'], 'off');
        }

        ## run the loop over array.
        foreach ($requestBag as $key => $value) {
            $this->saveDataInAppSettings($key, $value);
        }

        return 'OK';
    }

    /**
     * Purpose: this is used to get the user preference languages
     * */
    public function getUserPreferLocal()
    {

        $array = [];
        ## check multilingual option is enabled or not.
        $localeIsEnabled = $this->getDataFromAppSettings($this->constants::AppSettingsLabels['is_multilingual'], true);
        $array[$this->constants::AppSettingsLabels['is_multilingual']] = $localeIsEnabled;
        ## if multilingual option is enable
        $localeIsEnabled = $this->getDataFromAppSettings($this->constants::AppSettingsLabels['enabled_Languages']);
        $array         [$this->constants::AppSettingsLabels['enabled_Languages']] = $localeIsEnabled;

        return $array;
    }

    /**
     * ===================== END:: Save language settings.
     * */


    /**
     * ===================== General Methods
     * */

    /**
     * @param $error_collection
     * @param $error
     *
     * This function collect functions response and check if response of the function is in not equal to  'OK' then
     * save error in the bag.
     *
     * @return mixed
     */
    public function push_error($error_collection, $error)
    {
        if ($error <> 'OK') {
            array_push($error_collection, $error);
        }

        return $error_collection;
    }

    /**
     * Save setting in database.
     *
     * @param       $requestPost
     * @param array $requestFiles
     *
     * @return string|array
     */
    public function save_settings($requestPost, $requestFiles = [])
    {
        $response = [];

        ## post data
        foreach ($requestPost as $key => $value) {
            ## if key or value is empty then return back system to the loop.
            if (empty($key)) {
                continue;
            }

            ## save data in the appSetting table.
            $this->saveDataInAppSettings($key, $value);
        }

        ## upload files, image ex. school logo.
        foreach ($requestFiles as $key => $value) {
            $temp_array = [
                'fileArray' => $value,
                'allowedExtension' => ['jpg', 'jpeg', 'png', 'gif'],
                'public' => true,
                'path' => 'media'.DIRECTORY_SEPARATOR.'logos'.DIRECTORY_SEPARATOR,
                'returnPath' => true,
            ];

            ## upload file.
            $r_1 = $this->file_managment->uploadFile($temp_array);

            ## push error if exits.
            $response = $this->push_error($response, $r_1['response']);

            ## save array in it, to determine the base path of the file.

            ## save data in AppSetting table.
            $this->saveDataInAppSettings($key, $r_1['uploadedFileUrl']);
        }

        ## if response array is empty that's mean we've no error.
        if (empty($response)) {
            return 'OK';
        } else {
            return $response;
        }
    }

    /**
     * General Method save data in the app settings details.
     * This main function which is used to save data in app settings table.
     * This function validate the key & value according to the validation constraints.
     * */
    private function saveDataInAppSettings(?string $key, $value)
    {

        ## we are saving details in app setting details column, so if details are not in array format then convert it into array first.
        if (!is_array($value)) {
            $value = array($value);
        }

        ## before saving data in the app setting, check to find is this setting already exits or not.
        $appSettingsEntity = $this->app_settings_repository->findOneBy(['name' => $key]);
        ## if this settings already exits then update it.
        if ($appSettingsEntity) {
            ## update the app settings
            ## set the app settings name.
            $appSettingsEntity->setName($key);
            ## set the app settings details
            $appSettingsEntity->setDetails($value);
        } else {
            $appSettingsEntity = new \App\Entity\AppSettings();
            ## add new app settings.
            ## set the app settings name.
            $appSettingsEntity->setName($key);
            ## set the app settings details.
            $appSettingsEntity->setDetails($value);
        }

        ## save data to the the entity
        $this->entity_manager->persist($appSettingsEntity);
        ## flush it.
        $this->entity_manager->flush();
    }


    /**
     * @param string|null $key
     * @param bool|null $onlyOne
     *
     * @return \App\Entity\AppSettings|null
     */
    public function getDataFromAppSettings(?string $key, ?bool $onlyOne = false, $defaultValueIfEmpty = null)
    {
        ## before saving data in the app setting, check to find is this setting already exits or not.
        $response = $this->app_settings_repository->findOneBy(['name' => $key]);
        if ($response && $onlyOne) {
            $response = $response->getDetails()[0];
        }

        if (empty($response) && !empty($defaultValueIfEmpty)) {
            $response = $defaultValueIfEmpty;
        }

        return $response;
    }

    /**
     * This function will copy the app setting table in the session to prevent database requests
     * This function will call when user login
     */
    public function saveAppSettingsInSession()
    {
        try {
            $array = $this->app_settings_repository->InArray();
            $this->request->getCurrentRequest()->getSession()->set('app_settings', $array);
        } catch (\Exception $exception) {
            return $exception->getMessage();
        }
    }


    /**
     * @param string|null $index
     *
     * @return mixed|string
     * PURPOSE: This function is used to fetch app settings from the session instead of database.
     */
    public function getAppSettingsFromSessions(?string $index = null)
    {
        if (!$this->request->getCurrentRequest() instanceof Request) {
            return null;
        }

        $response = $this->request->getCurrentRequest()->getSession()->get('app_settings');

        ## if user request for the one app setting then return it.
        if (!empty($response) && !empty($index) && !empty($this->constants::AppSettingsLabels[$index])) {

            ## find the right value form the constants.
            $index = $this->constants::AppSettingsLabels[$index];
            if (!empty($response[$index])) {

                ## if settings is found in the database.
                $response = $response[$index];

                ## get the app settings details.
                if (count($response['details']) == 1 && isset($response['details'][0])) {
                    ## if app details have a single value the return it.
                    $response = $response['details'][0];
                } else {
                    ## extract app settings details.
                    $response = $response['details'];
                }
            } else {
                ## if settings not found in the database.
                $response = '';
            }
        } else {
            ## if settings not found in the database.
            $response = '';
        }

        ## if default language is empty then return english lang.
        if ($index == Constants::AppSettingsLabels['default_language'] && empty($response)) {
            $response = 'en';
        } else {
            if (($index == 'school_logo_path' || $index == 's_logo') && (empty($response))) {
                $response = Constants::FekaraLogoPath;
            }
        }

        return $response;
    }


    ## get the email_url
    public function get_email_url()
    {

    }

    /**
     * This method do some action based on the requested form.
     * */
    public function pre_validation()
    {
    }

    ## CheckList
    public function fekaraCheckList()
    {
        return [
            'payment_methods' => count($this->payment_methods_repository->findAll()) ? true : false,
            'account_heads' => count($this->account_heads_repository->findAll()) ? true : false,
            'add_fee' => count($this->student_fee_repository->findAll()) ? true : false,
            'add_subject' => count($this->subjects_repository->findAll()) > 0 ? true : false,
            'add_class' => count($this->class_template_repository->findAll()) > 0 ? true : false,
            'add_session' => count($this->session_repository->findAll()) > 0 ? true : false,
            'add_attendance_status' => (count($this->attendanceStatusRepository->findAll()) > 0) ? true : false,
            'add_accounts' => count($this->accountsRepository->findAll()) ? true : false,
            'schoolName' => $this->getDataFromAppSettings('s_name'),
            'classFeeScheduler' => count($this->classFeeScheduleRepository->findAll()) > 1 ? true : false,
        ];
    }

    /**
     *  Import database form Old Fekara to New Fekara.
     *
     * @throws DBALExceptionAlias
     */
    public function makeConnection()
    {
        $config = new \Doctrine\DBAL\Configuration();
        $connectionParams = array(
            'url' => 'mysql://'.$this->getAppSettingsFromSessions(
                    'remoteDatabaseUsername'
                ).':'.$this->getAppSettingsFromSessions('remoteDatabasePassword').'@'.$this->getAppSettingsFromSessions(
                    'remoteDatabaseHost'
                ).':'.$this->getAppSettingsFromSessions('remoteDatabasePort').'/'.$this->getAppSettingsFromSessions(
                    'remoteDatabaseName'
                ),
        );

        return \Doctrine\DBAL\DriverManager::getConnection($connectionParams, $config);
    }

    /**
     * @return bool|ConnectionAlias
     * Purpose: Check connection is established or not
     */
    public function isConnectionEstablished()
    {
        try {
            $response = $this->makeConnection();
        } catch (DBALExceptionAlias $e) {
            $response = false;
        }

        return $response;
    }

    /**
     *  Import Students and Guardian from old fekara
     *
     * @param UserPasswordEncoderInterface $userPasswordEncoder
     *
     * @param DefaultFunction $defaultFunction
     *
     * @return string
     * @throws DBALExceptionAlias
     */
    public function importStudentAndGuardians(
        UserPasswordEncoderInterface $userPasswordEncoder,
        DefaultFunction $defaultFunction
    ) {

        $connection = $this->makeConnection();
        ## List of students
        $query = "SELECT * FROM students ";
        $statement = $connection->prepare($query);
        $statement->execute();
        $studentList = $statement->fetchAll();
        foreach ($studentList as $key => $value) {

            // student is importing
            $array = (explode(' ', $value['name']));
            $userEntity = new User();
            $userEntity->setFName(!empty($array[0]) ? $array[0] : $value['name']);
            $userEntity->setLName(!empty($array[1]) ? $array[1] : '');
            $userEntity->setEmail($value['email']);
            $userEntity->setActiveBy(null);

            if ($value['gender'] == 'F') {
                $gender = 2;
            } else {
                if ($value['gender'] == 'M') {
                    $gender = 1;
                } else {
                    $gender = 3;
                }
            }

            $userEntity->setGender($gender);
            $userEntity->setCategory('s');
            $userEntity->setIsActive($value['active']);
            $userEntity->setAdmissionNumber((int)$value['roll']);
            $userEntity->setRoles(['ROLE_STUDENT']);
            $userEntity->setPassword('test');
            $userEntity->setIsDeleted($value['deleted']);
            $userEntity->setRegisterTime($defaultFunction->convert_datetimeStringIntoObject($value['admission_date']));
            $this->entity_manager->persist($userEntity);
            $this->entity_manager->flush();

            if (empty($userEntity->getAdmissionNumber()) || $userEntity->getAdmissionNumber() == 0) {
                $userEntity->setAdmissionNumber($userEntity->getId());
            }
            $userEntity->setPassword($userPasswordEncoder->encodePassword($userEntity, 'testing'));
            $this->entity_manager->persist($userEntity);
            $this->entity_manager->flush();


            $guardianN = explode(' ', $value['guardian_name']);

            //Guaridan is importing now...
            $guardian = new Guardian();
            $guardian->setPassword('testing');
            $guardian->setFname(!empty($guardianN[0]) ? $guardianN[0] : $value['guardian_name']);
            $guardian->setLname(!empty($guardianN[1]) ? $guardianN[1] : '');
            $guardian->setRoles(['ROLE_GUARDIAN']);
            $guardian->setIsActive($value['active']);
            $guardian->setIsDeleted($value['deleted']);

            $relation = 'g';
            $gender = 3;
            if ($relation == 'Father') {
                $relation = 'f';
                $gender = 1;
            } else {
                if ($relation == 'Mother') {
                    $relation = 'm';
                    $gender = 2;
                }
            }

            $guardian->setGender($gender);
            $guardian->setGuardianRelation($relation);
            $guardian->setCategory('g');
            $guardian->setEmail($value['guardian_email']);
            $guardian->setPhone($value['guardian_phone']);
            $guardian->addStudentReference($userEntity);

            $this->entity_manager->persist($guardian);
            $this->entity_manager->flush();

            $guardian->setPassword($userPasswordEncoder->encodePassword($guardian, 'testing'));
            $this->entity_manager->persist($guardian);
            $this->entity_manager->flush();
        }

        return 'OK';
    }


    /**
     * @param UserPasswordEncoderInterface $userPasswordEncoder
     * @param DefaultFunction $defaultFunction
     * PURPOSE: Import Teacher from old fekara as teacher in user table
     *
     * @return string
     */
    private function importTeacher(UserPasswordEncoderInterface $userPasswordEncoder, DefaultFunction $defaultFunction)
    {
        try {
            $connection = $this->makeConnection();
            $query = "SELECT * FROM teachers ";
            $statement = $connection->prepare($query);
            $statement->execute();
            $oldFekaraTeachersList = $statement->fetchAll();
            foreach ($oldFekaraTeachersList as $key => $value) {
                $array = (explode(' ', $value['name']));
                $userEntity = new User();
                $userEntity->setFName(!empty($array[0]) ? $array[0] : $value['name']);
                $userEntity->setLName(!empty($array[1]) ? $array[1] : '');
                $userEntity->setEmail($value['email']);
                $userEntity->setActiveBy(null);

                $gender = 3;

                $userEntity->setGender($gender);
                $userEntity->setCategory('t');
                $userEntity->setIsActive($value['active']);
                $userEntity->setAdmissionNumber(0);
                $userEntity->setRoles(['ROLE_TEACHER']);
                $userEntity->setPassword('test');
                $userEntity->setIsDeleted(false);
                $userEntity->setRegisterTime(
                    $defaultFunction->convert_datetimeStringIntoObject($value['date_of_joining'])
                );
                $this->entity_manager->persist($userEntity);
                $this->entity_manager->flush();

                if (empty($userEntity->getAdmissionNumber()) || $userEntity->getAdmissionNumber() == 0) {
                    $userEntity->setAdmissionNumber($userEntity->getId());
                }
                $userEntity->setPassword($userPasswordEncoder->encodePassword($userEntity, 'testing'));
                $this->entity_manager->persist($userEntity);
                $this->entity_manager->flush();
            }

            return 'OK';
        } catch (DBALExceptionAlias $e) {
            return $e->getMessage();
        }

    }


    /**
     * @param UserPasswordEncoderInterface $userPasswordEncoder
     * @return string
     */
    private function importAdmins(UserPasswordEncoderInterface $userPasswordEncoder)
    {
        try {
            $connection = $this->makeConnection();
            $query = "SELECT * FROM admins ";
            $statment = $connection->prepare($query);
            $statment->execute();
            $oldFekaraAdminsList = $statment->fetchAll();
            foreach ($oldFekaraAdminsList as $key => $value) {

                $array = (explode(' ', $value['name']));
                $userEntity = new User();
                $userEntity->setFName(!empty($array[0]) ? $array[0] : $value['name']);
                $userEntity->setLName(!empty($array[1]) ? $array[1] : '');
                $userEntity->setEmail($value['email']);
                $userEntity->setActiveBy(null);

                $gender = 3;

                $userEntity->setGender($gender);
                $userEntity->setCategory('a');
                $userEntity->setIsActive($value['active']);
                $userEntity->setAdmissionNumber(0);
                $userEntity->setRoles(['ROLE_ADMIN']);
                $userEntity->setPassword('test');
                $userEntity->setIsDeleted(false);
                $this->entity_manager->persist($userEntity);
                $this->entity_manager->flush();
                if (empty($userEntity->getAdmissionNumber()) || $userEntity->getAdmissionNumber() == 0) {
                    $userEntity->setAdmissionNumber($userEntity->getId());
                }
                $userEntity->setPassword($userPasswordEncoder->encodePassword($userEntity, 'testing'));
                $this->entity_manager->persist($userEntity);
                $this->entity_manager->flush();
            }
            $response = 'OK';
        } catch (DBALExceptionAlias $e) {
            $response = $e->getMessage();
        }

        return $response;
    }


    /**
     * @param ParameterBag $request
     * @param UserPasswordEncoderInterface $userPasswordEncoder
     *
     * @param DefaultFunction $defaultFunction
     *
     * @param SubjectService $subjectService
     * @param StudentClasses $studentClasses
     * @param StudentSession $studentSession
     * @param ClassTemplateRepository $classTemplateRepository
     * @param SubjectsRepository $subjectsRepository
     *
     * @return string|null
     * PURPOSE: Decide what to import table from database..
     */
    public function importDataFromOldFekara(
        ParameterBag $request,
        UserPasswordEncoderInterface $userPasswordEncoder,
        DefaultFunction $defaultFunction,
        SubjectService $subjectService,
        StudentClasses $studentClasses,
        StudentSession $studentSession,
        ClassTemplateRepository $classTemplateRepository,
        SubjectsRepository $subjectsRepository
    ) {
        ## check is connections is established or not .
        if (!$this->isConnectionEstablished()) {
            return 'Failed to connect with remote database';
        }
        $response = null;
        ## set execution time to infinite.
        ini_set('max_execution_time', '0');

        ## check what to import.
        switch ($request->get('whatTo')) {
            ## import student and guardian
            case 'importStudentAndGuardian':
                try {
                    $response = $this->importStudentAndGuardians($userPasswordEncoder, $defaultFunction);
                } catch (DBALExceptionAlias $e) {
                    $response = $e->getMessage();
                }
                break;
            ## Import Teacher
            case 'importTeacher':
                $response = $this->importTeacher($userPasswordEncoder, $defaultFunction);
                break;
            ## Import Admins
            case 'importAdmins':
                $response = $this->importAdmins($userPasswordEncoder);
                break;
            ## Import Subjects.
            case 'importSubjects';
                $response = $this->importSubjects($subjectService);
                break;
            ## Import Classes.
            case 'importClasses':
                $response = $this->importClass(
                    $studentClasses,
                    $studentSession,
                    $classTemplateRepository,
                    $subjectsRepository
                );
                break;
            case 'importSession':
                $response = $this->importSession($studentClasses, $studentSession, $classTemplateRepository);
                break;

        }

        ## return the response.
        return $response;
    }

    /**
     * @param SubjectService $subjectService
     * PURPOSE IMPORT Subject
     *
     * @return string
     */
    private function importSubjects(SubjectService $subjectService)
    {
        try {
            $connection = $this->makeConnection();
            $query = "SELECT * FROM subjects";
            $statment = $connection->prepare($query);
            $statment->execute();
            $subjects = $statment->fetchAll();
            foreach ($subjects as $key => $value) {
                $bag = new ParameterBag();
                $bag->set('subject_name', $value['subject']);
                $bag->set('subject_short_name', $value['short_name']);
                $subjectService->save__subject($bag);
            }

            $response = 'OK';
        } catch (DBALExceptionAlias $e) {
            $response = $e->getMessage();
        }

        return $response;
    }


    /**
     * @param StudentClasses $studentClasses
     * @param StudentSession $studentSession
     * @param ClassTemplateRepository $classTemplateRepository
     * @param SubjectsRepository $subjectsRepository
     * PURPOSE: Import Classes into Class_template
     *
     * @return string
     */
    private function importClass(
        StudentClasses $studentClasses,
        StudentSession $studentSession,
        ClassTemplateRepository $classTemplateRepository,
        SubjectsRepository $subjectsRepository
    ) {
        try {
            $connection = $this->makeConnection();

            $query = "SELECT * FROM batch_classes";
            $statement = $connection->prepare($query);
            $statement->execute();
            $classes = $statement->fetchAll();

            /** Session $value */
            foreach ($classes as $key => $value) {
                $bag2 = new ParameterBag();
                $bag2->set('name', $value['class_name']);
                $bag2->set('active', true);
                $bag2->set('fee', $value['fee']);
                $studentClasses->saveClasses($bag2, $studentSession);
            }

            $classTempalate = $classTemplateRepository->findAll();
            /** @var ClassTemplate $value */
            foreach ($classTempalate as $key => $value) {
                $query = "SELECT * FROM batch_classes where class_name = '{$value->getName()}'";
                $statment = $connection->prepare($query);
                $statment->execute();
                $classesFromOldFekeara = $statment->fetchAll();
                foreach ($classesFromOldFekeara as $key2 => $value2) {
                    $query = "SELECT * FROM `batch_classes_x_subjects` where class_id = '{$value2['id']}'";
                    $statment = $connection->prepare($query);
                    $statment->execute();
                    $classesFromOldFekeara = $statment->fetchAll();
                    foreach ($classesFromOldFekeara as $key3 => $value3) {
                        $query = "SELECT * FROM `subjects` where id = '{$value3['subject_id']}'";
                        $statment = $connection->prepare($query);
                        $statment->execute();
                        $classesFromOldFekeara___ = $statment->fetch();
                        $subjects__ = $subjectsRepository->findOneBy(
                            ['subject_name' => $classesFromOldFekeara___['subject']]
                        );

                        if ($subjects__ instanceof Subjects) {
                            $bag0909 = new ParameterBag();
                            $bag0909->set('___randsection', $value->getId());
                            $bag0909->set('name', $value->getName());
                            $bag0909->set('fee', $value->getFee());
                            $bag0909->set('active', true);
                            $bag0909->set('subjects', [$subjects__->getId()]);
                            $studentClasses->saveClasses($bag0909, $studentSession);
                        }
                    }
                }

            }

            $response = 'OK';
        } catch (DBALExceptionAlias $e) {
            $response = $e->getMessage();
        }

        return $response;
    }


    /**
     * @param StudentClasses $studentClasses
     * @param StudentSession $studentSession
     * @param ClassTemplateRepository $classTemplateRepository
     * PURPOSE: Import Sessions
     *
     * @return string
     */
    private function importSession(
        StudentClasses $studentClasses,
        StudentSession $studentSession,
        ClassTemplateRepository $classTemplateRepository
    ) {
        try {
            $connection = $this->makeConnection();
            $query = "SELECT * FROM batches";
            $statement = $connection->prepare($query);
            $statement->execute();
            $batches = $statement->fetchAll();
            foreach ($batches as $key => $value) {

                // saving batches
                $bag = new ParameterBag();
                $bag->set('name', trim($value['batch_name']));
                $bag->set('start_date', $value['start_date']);
                $bag->set('end_date', $value['end_date']);
                $bag->set('active', $value['active'] == 1 ? 'on' : 'off');
                $studentSession->saveSession($bag);
            }


            $query = "SELECT DISTINCT(class_name) FROM batch_classes";
            $statement = $connection->prepare($query);
            $statement->execute();
            $classes = $statement->fetchAll();
            foreach ($classes as $key => $value) {
                $query = "SELECT * FROM batch_classes where class_name = '{$value['class_name']}'";
                $statment = $connection->prepare($query);
                $statment->execute();
                $classes__ = $statment->fetchAll();
                foreach ($classes__ as $key2 => $value2) {
                    $query = "SELECT * FROM batches where id={$value2['batch_id']}";
                    $statment = $connection->prepare($query);
                    $statment->execute();
                    $batches = $statment->fetchAll();
                    foreach ($batches as $key3 => $value3) {
                        $fekaraSession = $this->session_repository->findOneBy(['name' => $value3['batch_name']]);
                        $fekaraClasses = $classTemplateRepository->findOneBy(['name' => $value['class_name']]);

                        if ($fekaraClasses instanceof ClassTemplate && $fekaraSession instanceof Session) {
                            $bag__ = new ParameterBag();
                            $bag__->set('___randsection', $fekaraClasses->getid());
                            $bag__->set('seSon', $fekaraSession->getId());
                            $bag__->set('name', $fekaraClasses->getName());
                            $bag__->set('fee', $fekaraClasses->getFee());
                            $studentClasses->saveClasses($bag__, $studentSession);
                        }
                    }
                }
            }
            $response = 'OK';
        } catch (DBALExceptionAlias $e) {
            $response = $e->getMessage();
        }

        return $response;
    }

}
