<?php

namespace App\Controller;

use App\Entity\Classes;
use App\Entity\ClassSessionEnrolment;
use App\Entity\ExamsClasses;
use App\Entity\ExamsClassesStudent;
use App\Entity\Session;
use App\Repository\AllocateTeacherToClassAndSubjectsRepository;
use App\Repository\ClassesRepository;
use App\Repository\ClassSessionEnrolmentRepository;
use App\Repository\ClassSubjectRepository;
use App\Repository\EmailTemplateEventsRepository;
use App\Repository\ExaminersRepository;
use App\Repository\ExamsClassesRepository;
use App\Repository\ExamsClassesStudentRepository;
use App\Repository\ExamsRepository;
use App\Repository\SessionRepository;
use App\Repository\SubjectsRepository;
use App\Repository\UserRepository;
use App\Service\AppSettings;
use App\Service\DefaultFunction;
use App\Service\Exams\ExamsService;
use App\Service\Guardian\GuardianService;
use App\Service\SMSGateways\OperationalionalGatewayClass;
use App\Service\TriggerNotifications;
use Knp\Bundle\SnappyBundle\Snappy\Response\PdfResponse;
use Knp\Snappy\Pdf;
use Nzo\UrlEncryptorBundle\Annotations\ParamDecryptor;
use Nzo\UrlEncryptorBundle\UrlEncryptor\UrlEncryptor;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\BinaryFileResponse;
use Symfony\Component\HttpFoundation\HeaderUtils;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\Translation\TranslatorInterface;

/**
 * @IsGranted("ROLE_USER")
 * */
class ExaminationController extends AbstractController
{

    /**
     * @var ExamsService
     */
    private $exams_service;
    /**
     * @var ExamsRepository
     */
    private $exams_repository;
    /**
     * @var ExaminersRepository
     */
    private $examiners_repository;
    /**
     * @var ExamsClassesRepository
     */
    private $exams_classes_repository;
    /**
     * @var ExamsClassesStudentRepository
     */
    private $exams_classes_student_repository;
    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var SubjectsRepository
     */
    private $subjects_repository;

    /**
     * @var ClassesRepository
     */
    private $classes_repository;
    /**
     * @var SessionRepository
     */
    private $session_repository;
    /**
     * @var ClassSubjectRepository
     */
    private $class_subject_repository;
    /**
     * @var UrlEncryptor
     */
    private $url_encryptor;
    /**
     * @var Pdf
     */
    private $pdf;
    /**
     * @var AppSettings
     */
    private $appSettings;

    /**
     * ExaminationController constructor.
     *
     * @param UrlEncryptor $url_encryptor
     * @param ClassSubjectRepository $class_subject_repository
     * @param SessionRepository $session_repository
     * @param ExamsService $exams_service
     * @param SubjectsRepository $subjects_repository
     * @param UserRepository $user_repository
     * @param ExamsRepository $exams_repository
     * @param ExaminersRepository $examiners_repository
     * @param ExamsClassesRepository $exams_classes_repository
     * @param ExamsClassesStudentRepository $exams_classes_student_repository
     * @param ClassesRepository $classes_repository
     * @param Pdf $pdf
     * @param AppSettings $appSettings
     */
    public function __construct(
        UrlEncryptor $url_encryptor,
        ClassSubjectRepository $class_subject_repository,
        SessionRepository $session_repository,
        ExamsService $exams_service,
        SubjectsRepository $subjects_repository,
        UserRepository $user_repository,
        ExamsRepository $exams_repository,
        ExaminersRepository $examiners_repository,
        ExamsClassesRepository $exams_classes_repository,
        ExamsClassesStudentRepository $exams_classes_student_repository,
        ClassesRepository $classes_repository,
        Pdf $pdf,
        AppSettings $appSettings
    ) {
        $this->exams_service = $exams_service;
        $this->exams_repository = $exams_repository;
        $this->examiners_repository = $examiners_repository;
        $this->exams_classes_repository = $exams_classes_repository;
        $this->exams_classes_student_repository = $exams_classes_student_repository;
        $this->user_repository = $user_repository;
        $this->subjects_repository = $subjects_repository;
        $this->classes_repository = $classes_repository;
        $this->session_repository = $session_repository;
        $this->class_subject_repository = $class_subject_repository;
        $this->url_encryptor = $url_encryptor;
        $this->pdf = $pdf;
        $this->appSettings = $appSettings;
    }

    /**
     * @Route("{_locale}/exams/list", name="exam_list")
     * @IsGranted("ROLE_EXAMINATION_MANAGER")
     */
    public function exams()
    {
        return $this->render(
            'examination/exams.html.twig',
            [
                'exams' => $this->exams_repository->findAll(),
            ]
        );
    }

    /**
     * @Route("{_locale}/examiner/list", name="examiner_list")
     * @IsGranted("ROLE_EXAMINATION_MANAGER")
     */
    public function examiner()
    {
        return $this->render(
            'examination/examiners.html.twig',
            [
                'examiner' => $this->examiners_repository->findAll(),
                't___l' => $this->user_repository->findBy(['category' => 't']),
                'examClass' => $this->exams_classes_repository->findAll(),
            ]
        );
    }


    /**
     * @param TranslatorInterface $translator
     * ## role manager.
     */
    public function roleManager(TranslatorInterface $translator)
    {
        ## check both roles to the user.
        if (!$this->isGranted('ROLE_EXAMINATION_MANAGER') && !$this->isGranted('ROLE_TEACHER')) {
            $this->denyAccessUnlessGranted(
                'ROLE_TEACHER',
                null,
                $translator->trans('Access Denied, You must have Teacher Role')
            );
            $this->denyAccessUnlessGranted(
                'ROLE_EXAMINATION_MANAGER',
                null,
                $translator->trans('Access Denied, You must have Examination Manager Role')
            );
        }
    }

    /**
     * @param TranslatorInterface $translator
     * ## role checker.
     */
    public function roleChecker(TranslatorInterface $translator)
    {
        ## check both roles to the user.
        if (!$this->isGranted('ROLE_EXAMINATION_MANAGER') && !$this->isGranted('ROLE_TEACHER')) {
            $this->denyAccessUnlessGranted(
                'ROLE_TEACHER',
                null,
                $translator->trans('Access Denied, You must have Teacher Role')
            );
            $this->denyAccessUnlessGranted(
                'ROLE_EXAMINATION_MANAGER',
                null,
                $translator->trans('Access Denied, You must have Examination Manager Role')
            );
        }
    }

    /**
     * Exam Schedule
     * */


    /**
     * @Route("{_locale}/exam/make-exam-schedule/", name="exam_schedule")
     * PURPOSE: add exam schedule.
     *
     * @param TranslatorInterface $translator
     * @param Request $request
     *
     * @return Response
     */
    public function exams_schedule(TranslatorInterface $translator, Request $request)
    {

        $examsClass = new ExamsClasses();
        $examsClassSelectedSubjects = null;
        if (!empty($this->url_encryptor->decrypt($request->query->get('id')))) {
            $examsClass = $this->exams_service->get__examClass(
                $this->url_encryptor->decrypt($request->query->get('id'))
            );
            if ($examsClass instanceof ExamsClasses) {
                ## get the class ref from the exam class
                $classID = $examsClass->getClass()->getId();
                ## get the subject from the reference of class.
                $examsClassSelectedSubjects = $this->class_subject_repository->findBy(['classes__' => $classID]);
            } else {
                $examsClass = new ExamsClasses();
            }
        }

        $this->roleManager($translator);

        return $this->render(
            'examination/examSchedule/exams_schedule.html.twig',
            [
                'examsClass' => $examsClass,
                'examsClassSelectedSubjects' => $examsClassSelectedSubjects,
                ## exam list
                'exam' => $this->exams_repository->findAll(),
                ## Class list
                'class__list' => $this->classes_repository->getActiveClassesOfActiveSession(),
                ## Teacher name
                'teacherName' => $this->user_repository->findBy(['category' => TeacherController::userCategory]),
            ]
        );
    }

    /**
     * @Route("{_locale}/exam/exam-schedule/list", name="exam_schedule_list")
     * PURPOSE: add exam schedule.
     *
     * @param Request $request
     *
     * @param TranslatorInterface $translator
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    public function exams_schedule_list(
        Request $request,
        TranslatorInterface $translator,
        DefaultFunction $defaultFunction,
        AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
    ) {

        ## check both roles to the user.
        $this->roleManager($translator);

        $Bag = new ParameterBag();
        $Bag->set('se', $this->url_encryptor->decrypt($request->get('session')));
        $Bag->set('cl', $this->url_encryptor->decrypt($request->get('class')));
        $Bag->set('su', $this->url_encryptor->decrypt($request->get('subject')));
        $Bag->set('eD', $request->get('examDate'));
        $Bag->set('loc', $request->get('location'));
        $Bag->set('ex', $this->url_encryptor->decrypt($request->get('examiner')));
        ## search by exam name
        $Bag->set('exam', $this->url_encryptor->decrypt($request->get('exams')));

        if ($this->getUser()->getCategory() === 't') {
            $Bag->set(
                'classes',
                $allocateTeacherToClassAndSubjectsRepository->getUniqueClassAllocationByTeacherID(
                    $this->getUser()->getId()
                )
            );
        }

        $sendingData = [
            ## exam schedule list
            'examSchedule' => $this->exams_classes_repository->examClassAdvanceSearch($Bag),
            ## exams list
            'exams' => $this->exams_repository->findAll(),
            ## sessions
            'sessions' => $this->session_repository->findAll(),
            ## Class
            'classes' => $this->classes_repository->findAll(),
            ## Subject
            'subjects' => $this->class_subject_repository->findAll(),
            ## Examiners
            'examiners' => $this->user_repository->findBy(['category' => TeacherController::userCategory]),
            ## exam locations
            'locations' => $this->exams_classes_repository->distinctExamClassLocation(),
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $filename = 'Exam Schedule List';
            $templateName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('examschedulelist')
            );
            if (empty($templateName)) {
                $templateName = 'pdfPrints/exam.root/exam.schedule.list/default.html.twig';
            }

            $sendingData['examDetail'] = $this->exams_repository->find($Bag->get('exam'));
            $sendingData['sessionDetail'] = $this->session_repository->find($Bag->get('se'));
            $sendingData['classDetail'] = $this->classes_repository->find($Bag->get('cl'));
            $sendingData['subjectDetail'] = $this->class_subject_repository->find($Bag->get('su'));
            $sendingData['examinerDetail'] = $this->examiners_repository->find($Bag->get('ex'));
            $sendingData['locationDetail'] = $Bag->get('loc');

            $html_fe = $this->renderView($templateName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }


        return $this->render('examination/examSchedule/examScheduleList.html.twig', $sendingData);
    }


    /**
     * @Route("{_locale}/student/exam/exam-schedule/list", name="exam_schedule_list_student")
     * PURPOSE: add exam schedule.
     *
     * @param Request $request
     * @IsGranted("ROLE_STUDENT")
     * @param TranslatorInterface $translator
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    public function exams_schedule_listStudentCommingExam(
        Request $request,
        TranslatorInterface $translator,
        DefaultFunction $defaultFunction,
        ClassSessionEnrolmentRepository $classSessionEnrolmentRepository
    ) {


        $lastEnrolmentOfStudent = $classSessionEnrolmentRepository->getLastEnrolledClassOUser(
            $this->getUser()->getId()
        );
        if (!empty($lastEnrolmentOfStudent[0])) {
            $lastEnrolmentOfStudent = $lastEnrolmentOfStudent[0];
        }

        $class = null;
        if ($lastEnrolmentOfStudent instanceof ClassSessionEnrolment) {
            $class = $lastEnrolmentOfStudent->getClassSession()->getId();
        }
        $Bag = new ParameterBag();
        $Bag->set('cl', $class);

        $sendingData = [
            ## exam schedule list
            'examSchedule' => $this->exams_classes_repository->examClassAdvanceSearch($Bag),
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $filename = 'Exam Schedule List';
            $templateName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('examschedulelist')
            );
            if (empty($templateName)) {
                $templateName = 'pdfPrints/exam.root/exam.schedule.list/default.html.twig';
            }

            $sendingData['examDetail'] = $this->exams_repository->find($Bag->get('exam'));
            $sendingData['sessionDetail'] = $this->session_repository->find($Bag->get('se'));
            $sendingData['classDetail'] = $this->classes_repository->find($Bag->get('cl'));
            $sendingData['subjectDetail'] = $this->class_subject_repository->find($Bag->get('su'));
            $sendingData['examinerDetail'] = $this->examiners_repository->find($Bag->get('ex'));
            $sendingData['locationDetail'] = $Bag->get('loc');

            $html_fe = $this->renderView($templateName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        return $this->render('examination/examSchedule/student/examScheduleList.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/guardian/exam/exam-report/list", name="exam_report_list_guardian")
     * PURPOSE: add exam schedule.
     *
     * @param Request $request
     * @IsGranted("ROLE_GUARDIAN")
     * @param TranslatorInterface $translator
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    public function exams_schedule_listGuardian(
        Request $request,
        UrlEncryptor $urlEncryptor,
        DefaultFunction $defaultFunction,
        $id = null,
        $student = null
    ) {

        $Bag = new ParameterBag();
        ## student based.
        $Bag->set('student__based', $urlEncryptor->decrypt($request->query->get('srf')));
        ## search attendance wise duration.
        $Bag->set('exam_markingDate', $request->query->get('date'));
        $Bag->set('userArray', GuardianService::getAttachedStudentsIdWithGuardian($this->getUser()));

        $sendingData = [
            'ExamClassStudent' => $this->exams_classes_student_repository->advanceSearch($Bag),
            'studentName' => $urlEncryptor->decrypt($request->query->get('student')),
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $filename = 'Student base Exam Report';

            ## find the template in the database
            $templatePath = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('ExamStudentBasedReporting')
            );
            ## if template is not found in the database then use default template
            if (empty($templatePath)) {
                $templatePath = 'pdfPrints/exam.root/Exam.Student.Based.Reporting/default.html.twig';
            }

            ## student details
            $sendingData['studentDetail'] = $this->user_repository->find($Bag->get('student__based'));
            $sendingData['date'] = $Bag->get('exam_markingDate');

            ## compiling the template
            $html_fe = $this->renderView($templatePath, $sendingData);

            ## send the response.
            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        return $this->render('examination/reporting/guardian/student_base_reporting.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/student/exam/exam-report/list", name="exam_report_list_student")
     * PURPOSE: add exam schedule.
     *
     * @param Request $request
     * @IsGranted("ROLE_STUDENT")
     * @param TranslatorInterface $translator
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    public function exams_schedule_listStudent(
        Request $request,
        UrlEncryptor $urlEncryptor,
        DefaultFunction $defaultFunction,
        $id = null,
        $student = null
    ) {

        $Bag = new ParameterBag();
        ## student based.
        $Bag->set('student__based', $this->getUser()->getId());

        $sendingData = [
            'ExamClassStudent' => $this->exams_classes_student_repository->advanceSearch($Bag),
            'studentName' => $this->getUser()->getFullName(),
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $filename = 'Student base Exam Report';

            ## find the template in the database
            $templatePath = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('ExamStudentBasedReporting')
            );
            ## if template is not found in the database then use default template
            if (empty($templatePath)) {
                $templatePath = 'pdfPrints/exam.root/Exam.Student.Based.Reporting/default.html.twig';
            }

            ## student details
            $sendingData['studentDetail'] = $this->user_repository->find($Bag->get('student__based'));
            $sendingData['date'] = $Bag->get('exam_markingDate');

            ## compiling the template
            $html_fe = $this->renderView($templatePath, $sendingData);

            ## send the response.
            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        return $this->render('examination/reporting/student/student_base_reporting.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/exam/mark-exam-sheet/", name="marks__exam_sheet")
     * @param TranslatorInterface $translator
     *
     * @return Response
     */
    public function mark_exam_sheet(TranslatorInterface $translator)
    {

        $this->roleChecker($translator);

        ## check both roles to the user.
        if (!$this->isGranted('ROLE_EXAMINATION_CHECKER') && !$this->isGranted('ROLE_TEACHER')) {
            $this->denyAccessUnlessGranted(
                'ROLE_TEACHER',
                null,
                $translator->trans('Access Denied, You must have Teacher Role')
            );
            $this->denyAccessUnlessGranted(
                'ROLE_EXAMINATION_CHECKER',
                null,
                $translator->trans('Access Denied, You must have Examination Checker Role')
            );
        }

        return $this->render(
            'examination/marks_exams.html.twig',
            [
                'examsMarkingResult' => $this->exams_classes_student_repository->findAll(),
                'examSchedule' => $this->exams_classes_repository->findAll(),
                'class__list' => $this->classes_repository->findAll(),
                's___l' => $this->user_repository->findBy(['category' => 's']),
            ]
        );
    }

    /**
     * @Route("{_locale}/exam/mark-exam-sheet/enter-marks/{ClassID}/{examClassID}/{sendMessage}", name="marks__exam_sheet_enter_marks",
     *                                                                               defaults={"ClassID":null,
     *                                                                               "examClassID": null}) PURPOSE:
     *                                                                               Enter passing marks of students
     * @ParamDecryptor(params={"ClassID"})
     * @ParamDecryptor(params={"examClassID"})
     *
     * @param                                 $ClassID
     * @param                                 $examClassID
     * @param ClassSessionEnrolmentRepository $class_session_enrolment_repository
     *
     * @param TranslatorInterface $translator
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     *
     * @param null $sendMessage
     * @param OperationalionalGatewayClass $operationalionalGatewayClass
     * @param EmailTemplateEventsRepository $emailTemplateEventsRepository
     * @return Response
     */
    public function enteringPassingMarksOfStudents(
        $ClassID,
        $examClassID,
        ClassSessionEnrolmentRepository $class_session_enrolment_repository,
        TranslatorInterface $translator,
        Request $request,
        DefaultFunction $defaultFunction,
        $sendMessage = null,
        EmailTemplateEventsRepository $emailTemplateEventsRepository,
        TriggerNotifications $triggerNotifications
    ) {
        $this->roleChecker($translator);

        $sendingData = [
            'examsClass' => $this->exams_classes_repository->markExamsResults($examClassID),
            'listOfStudentsWhoAreEnrolInClass' => $class_session_enrolment_repository->findBy(
                ['class_session' => $ClassID]
            ),
            'examClass' => $this->exams_classes_repository->find($examClassID),
            'classID' => $ClassID,
            'examClassID' => $examClassID,
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $filename = 'Exam sheet';
            $templateName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('StudentEnrolmentInClassAndSubjects')
            );
            if (empty($templateName)) {
                $templateName = 'pdfPrints/exam.root/Student.Enrolment.In.Class.And.Subjects/default.html.twig';
            }
            $html_fe = $this->renderView($templateName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }
        if ($sendMessage == 1) {

            foreach ($sendingData['examClass']->getExamsClassesStudents()->toArray() as $noInMarks) {
                /** @var ExamsClassesStudent $noInMarks */
                $noInMarks->sendSMSAboutGainMarks(
                    $sendingData['examClass'],
                    $triggerNotifications
                );
            }

            return $this->redirectToRoute(
                'marks__exam_sheet_enter_marks',
                [
                    'ClassID' => $this->url_encryptor->encrypt($request->attributes->get('ClassID')),
                    'examClassID' => $this->url_encryptor->encrypt($request->attributes->get('examClassID')),
                ]
            );

        }

//        dd($sendingData['examClass']);

//        dd($sendingData['listOfStudentsWhoAreEnrolInClass']);


        return $this->render('examination/EnterMarks/entering_passing_marks.html.twig', $sendingData);
    }


    /**
     *  =============== Reporting
     * */

    /**
     * @Route("{_locale}/exam/mark-based-reporting/{id}", name="exam_based_reporting", defaults={"id"=null})
     * @ParamDecryptor(params={"id"})
     */
    public function examBaseReporting($id = null)
    {

        ## get the exams Class.
        $examClass = $this->exams_service->get__examClass($id);
        ## if exam not found
        if (!$examClass instanceof ExamsClasses) {
            return $this->redirectToRoute('exam_schedule');
        }

        return $this->render(
            'examination/reporting/exam_name_based_reporting.html.twig',
            [
                'ExamClass' => $examClass,
                'ExamClassStudent' => $this->exams_classes_student_repository->findBy(
                    ['exams_classes' => $examClass->getId()]
                ),
            ]
        );
    }

    /**
     * @Route("{_locale}/exam/student-based-reporting", name="student_based_reporting",
     *                                                                      defaults={"id"=null, "student"=null})
     * @ParamDecryptor(params={"id","student"})
     * @param Request $request
     * @param UrlEncryptor $urlEncryptor
     * @param DefaultFunction $defaultFunction
     * @param null $id
     * @param null $student
     *
     * @param TriggerNotifications $triggerNotifications
     * @return Response
     */
    public function studentBaseReporting(
        Request $request,
        UrlEncryptor $urlEncryptor,
        DefaultFunction $defaultFunction,
        $id = null,
        $student = null,
        TriggerNotifications $triggerNotifications
    ) {

        $Bag = new ParameterBag();
        ## student based.
        $Bag->set('student__based', $urlEncryptor->decrypt($request->query->get('srf')));
        ## search attendance wise duration.
        $Bag->set('exam_markingDate', $request->query->get('date'));

        $sendingData = [
            'ExamClassStudent' => $this->exams_classes_student_repository->advanceSearch($Bag),
            'studentName' => $urlEncryptor->decrypt($request->query->get('student')),
            'listOfStudents' => $this->user_repository->findAll(),
        ];

        ## print pdf file
        if ($request->query->get('printResult') == 1) {
            $filename = 'Student base Exam Report';

            ## find the template in the database
            $templatePath = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('ExamStudentBasedReporting')
            );
            ## if template is not found in the database then use default template
            if (empty($templatePath)) {
                $templatePath = 'pdfPrints/exam.root/Exam.Student.Based.Reporting/default.html.twig';
            }

            ## student details
            $sendingData['studentDetail'] = $this->user_repository->find($Bag->get('student__based'));
            $sendingData['date'] = $Bag->get('exam_markingDate');

            ## compiling the template
            $html_fe = $this->renderView($templatePath, $sendingData);

            ## send the response.
            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        ## send Message..
        if ($request->query->get('sendMessage') == 1) {
            foreach ($sendingData['ExamClassStudent'] as $examClassStudent) {
                /** @var ExamsClassesStudent $examClassStudent */
                $examClassStudent->sendSMSAboutGainMarks($examClassStudent->getExamsClasses(), $triggerNotifications);
            }

            return $this->redirectToRoute(
                'student_based_reporting',
                [
                    'srf' => $request->query->get('srf'),
                    'date' => $request->query->get('date'),
                ]
            );
        }

        ## send only Email to garudain
        if ($request->query->get('sendOnlyEmail') == 1) {

            $examClassStudent = new ExamsClassesStudent();
            $examClassStudent->sendOnlyEmailAboutExamReport(
                $triggerNotifications,
                $sendingData['ExamClassStudent'],
                $this->user_repository->find($urlEncryptor->decrypt($request->query->get('srf')))
            );

            return $this->redirectToRoute(
                'student_based_reporting',
                [
                    'srf' => $request->query->get('srf'),
                    'date' => $request->query->get('date'),
                ]
            );
        }


        return $this->render('examination/reporting/student_base_reporting.html.twig', $sendingData);
    }


    /**
     * @Route("{_locale}/exam/class-subject-based-reporting/", name="classBasedMarkReporting")
     * @param ClassSessionEnrolmentRepository $classSessionEnrolmentRepository
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     *
     * @return Response
     */
    public function classAndSubjectBasedReporting(
        ClassSessionEnrolmentRepository $classSessionEnrolmentRepository,
        Request $request,
        DefaultFunction $defaultFunction
    ) {
        $classID = $request->query->get('classId', null);
        $examID = $request->query->get('examId', null);

        ## data that we need to send to the file.
        $sendingData = [
            'classList' => $this->classes_repository->findAll(),
            'enrolledStudentsInClass' => $classSessionEnrolmentRepository->findBy(['class_session' => $classID]),
            'subjectsInClasses' => $this->class_subject_repository->findBy(['classes__' => $classID]),
            'examsClassStudent' => $this->exams_classes_repository->findBy(
                [
                    'class' => $classID,
                    'exam' => $examID,
                ]
            ),
            'examsClass' => $this->exams_classes_repository->getTheScheduledExamsListByRefOfClass($classID, $examID),
            'exams' => $this->exams_repository->findAll(),
        ];

        if (!empty($classID)) {
            $sendingData['classDetails']
                = $this->classes_repository->find($classID);
        }

        ## print pdf report
        if ($request->query->get('printResult') == 1) {
            ## show class and exam name in the report

            $sendingData['class'] = null;
            if ($classID) {
                $sendingData['class'] = $this->classes_repository->find($classID);
            }
            $sendingData['exam'] = null;
            if ($examID) {
                $sendingData['exam'] = $this->exams_repository->find($examID);
            }

            $filename = 'Exam Report';

            $reportFileName = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('examClassBasedReporting')
            );
            if (empty($reportFileName)) {
                $reportFileName = 'pdfPrints/exam.root/exam.Class.Based.Reporting/default.html.twig';
            }

            $html_fe = $this->renderView($reportFileName, $sendingData);

            return new Response(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        return $this->render('examination/reporting/classBasedReporting.html.twig', $sendingData);

    }


    /**
     * @Route("{_locale}/examination/requestHandler/}", name="requestHandler_exams_student")
     * @param Request $request
     *
     * @return JsonResponse
     */
    public function ajaxHandler(Request $request)
    {
        ## if user is not logged in the redirect th e use rto login page.
        $response = 'OK';

        if (empty($query)) {
            ## get the query variable from the get request.
            $query = $request->query->get('query');
        }

        switch ($query) {
            case 'ex__m':
                ## add - edit exams
                $response = $this->exams_service->save__exams($request->request);
                break;
            case 'ex__mine':
                ## add - edit examiner
                // @deprecated because we're using examiner assigning from Exam Schedule
                //$response = $this->exams_service->save__examiner( $request->request );
                break;
            case 'ex__cls':
                ## add - edit exam classes (making exam schedule)
                $response = $this->exams_service->save__exam_schedule($request->request);
                break;
            case 'ex__clsStud':
                ## add - edit exam classes (making exam passing marks)
                $response = $this->exams_service->save__examsClassesStudent($request->request);
                break;
            case 'enterMarksOfStudents':
                ## enter marks of students
                $response = $this->exams_service->save__bulkExamClassesStudent($request->request);
                break;
        }

        return new JsonResponse($response, 200);
    }

}
