<?php

namespace App\Security;

use App\Entity\User;
use App\Repository\GuardianRepository;
use App\Repository\UserRepository;
use App\Service\AppSettings;
use App\Service\CustomTokenManager;
use Doctrine\DBAL\Types\TextType;
use Doctrine\ORM\EntityManager;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\Driver\DatabaseDriver;
use Facebook\WebDriver\Exception\NoAlertOpenException;
use Symfony\Bundle\FrameworkBundle\Controller\ControllerTrait;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\HttpFoundation\Session\SessionInterface;
use Symfony\Component\HttpKernel\Event\PostResponseEvent;
use Symfony\Component\HttpKernel\Exception\MethodNotAllowedHttpException;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Routing\Exception\MethodNotAllowedException;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Encoder\UserPasswordEncoderInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Core\Exception\CustomUserMessageAuthenticationException;
use Symfony\Component\Security\Core\Exception\InvalidCsrfTokenException;
use Symfony\Component\Security\Core\Exception\UnsupportedUserException;
use Symfony\Component\Security\Core\Exception\UsernameNotFoundException;
use Symfony\Component\Security\Core\Security;
use Symfony\Component\Security\Core\User\UserInterface;
use Symfony\Component\Security\Core\User\UserProviderInterface;
use Symfony\Component\Security\Csrf\CsrfToken;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Component\Security\Guard\AbstractGuardAuthenticator;
use Symfony\Component\Security\Guard\Authenticator\AbstractFormLoginAuthenticator;
use Symfony\Component\Security\Http\Util\TargetPathTrait;
use Symfony\Component\Validator\Constraints\Json;
use Twig\Sandbox\SecurityNotAllowedFilterError;

class UserLoginAuthenticator extends AbstractFormLoginAuthenticator
{

    use TargetPathTrait;

    private $urlGenerator;
    private $csrfTokenManager;
    private $passwordEncoder;
    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var Session
     */
    private $session;
    /**
     * @var AppSettings
     */
    private $app_settings;
    /**
     * @var GuardianRepository
     */
    private $guardian_repository;
    /*
     * request come from where??
     * */
    private $request_from_route;
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;
    /**
     * @var CustomTokenManager
     */
    private $customTokenManager;

    private $stateLessAuth = false;
    private $generatedToken = null;

    public function __construct(
        CsrfTokenManagerInterface $csrfTokenManager,
        UserPasswordEncoderInterface $passwordEncoder,
        UrlGeneratorInterface $urlGenerator,
        UserRepository $user_repository,
        SessionInterface $session,
        AppSettings $app_settings,
        GuardianRepository $guardian_repository,
        EntityManagerInterface $entityManager,
        CustomTokenManager $customTokenManager
    ) {
        $this->csrfTokenManager = $csrfTokenManager;
        $this->passwordEncoder = $passwordEncoder;
        $this->urlGenerator = $urlGenerator;
        $this->user_repository = $user_repository;
        $this->session = $session;
        $this->app_settings = $app_settings;
        $this->guardian_repository = $guardian_repository;
        $this->entityManager = $entityManager;
        $this->customTokenManager = $customTokenManager;
    }

    /**
     * this method will check, current request is login request or not.
     * @param Request $request
     * @return bool
     */
    public
    function supports(
        Request $request
    ) {
        return ('user_login' === $request->attributes->get('_route') || 'guardian_login' === $request->attributes->get(
                    '_route'
                )) && ($request->isMethod('POST'));
    }

    /**
     * @param Request $request
     *
     * @return array|bool|mixed
     */
    public
    function getCredentials(
        Request $request
    ) {

        ## first get error message.
        $this->session->getFlashBag()->get('LoginError');
        ## request from normal user or from the guardian.
        $this->request_from_route = $request->attributes->get('_route');

        ## get the form data & save in the array.
        //$FormDataAr = $request->request->get( 'user_login_form' );
        $FormDataAr['authBy'] = $request->request->get('authBy');
        $FormDataAr['password'] = $request->request->get('password');
        $FormDataAr['csrf_token'] = $request->request->get('_token_csrf_');
        $FormDataAr['client_id'] = $request->request->get('client_id');
        ## check user's email, password is not empty
        if (empty($FormDataAr['authBy'])) {
            ## add error message in a flash
            $this->session->getFlashBag()->add('LoginError', 'Please type email or Admission Number');

            return false;
        } else {
            if (empty($FormDataAr['password'])) {
                ## add message in a session
                $this->session->getFlashBag()->add('LoginError', 'Please type password');

                return false;
            }
        }
        ## get the authby value and send back to the contorller.
        $request->getSession()->set(Security::LAST_USERNAME, $FormDataAr['authBy']);


        ## make an data array & send to the other method.
        $credentials = [
            'authBy' => $FormDataAr['authBy'],
            'password' => $FormDataAr['password'],
            'csrf_token' => $FormDataAr['csrf_token'],
            'client_id' => $FormDataAr['client_id'],
        ];

        return $credentials;
    }

    /**
     * @param mixed $credentials
     * @param UserProviderInterface $userProvider
     *
     * @return User|UserInterface|null
     */
    public
    function getUser(
        $credentials,
        UserProviderInterface $userProvider
    ) {


        if (empty($credentials['client_id'])) {
            ## for the state-full authentication
            $token = new CsrfToken(getenv('APP_SECRET'), $credentials['csrf_token']);
            ## validate CSRF token.
            if (!$this->csrfTokenManager->isTokenValid($token)) {
                throw new InvalidCsrfTokenException();
            }
        } else {
            ## for stateless authentication.
            if ($credentials['client_id'] <> getenv('DESKTOP_CLIENT_ID')) {
                throw new InvalidCsrfTokenException();
            }
            ## this is stateless authentication
            $this->stateLessAuth = true;
        }

        ## to prevent errors.
        $user = false;

        if ($this->request_from_route == 'user_login') {
            ## if normal user want to login in the application.
            ## check if authBy is email then check for email
            if (filter_var($credentials['authBy'], FILTER_VALIDATE_EMAIL)) {
                ## find user, by email
                $user = $this->user_repository->findOneBy(['email' => $credentials['authBy']]);
            } else {
                ## if not found a user regarding email then we will search by the admission number.
                $user = $this->user_repository->findOneBy(['admission_number' => $credentials['authBy']]);
            }

        } else {
            if ($this->request_from_route == 'guardian_login') {
                ## if guardian want to login.
                $user = $this->guardian_repository->findOneBy(['guardian_email' => $credentials['authBy']]);
            }
        }
        if (!$user) {
            ## if still not found a user with the admission number then we will return an exception that user not found
            throw new CustomUserMessageAuthenticationException(sprintf("%s, user not found", $credentials['authBy']));
        } else {
            ## is active check only for the normal user, teacher, student and for staff.
            if ($this->request_from_route == 'user_login') {

                ## if we found a user then we'll check is user is deleted or not.
                if ($user->getIsDeleted()) {
                    ## if user is deleted  then thrown an error.
                    throw new CustomUserMessageAuthenticationException(
                        sprintf("%s, user is deleted", $credentials['authBy'])
                    );
                }

                ## if we found a user then we'll check is user is active or not.
                if (!$user->getIsActive()) {
                    ## if still not found a user with the admission number then we will return an exception that user not found
                    throw new CustomUserMessageAuthenticationException(
                        sprintf("%s, user is deactivated", $credentials['authBy'])
                    );
                }
            }
        }

        return $user;
    }

    /**
     * @param mixed $credentials
     * @param UserInterface $user
     *
     * @return bool
     */
    public
    function checkCredentials(
        $credentials,
        UserInterface $user
    ) {


        $passwordValidation = $this->passwordEncoder->isPasswordValid($user, $credentials['password']);
        if ($passwordValidation) {
            ## get the user object.
            $loggedInUser = $this->user_repository->find($user->getid());
            ## if password validate then update the token - api token
            try {
                $this->generatedToken = $this->customTokenManager->saveUpdateApiToken($loggedInUser, null);
                if (substr($this->generatedToken, 0, 2) == 'OK') {
                    $this->generatedToken = substr($this->generatedToken, 2);
                } else {
                    $this->generatedToken = 'Token is not generated, Please contact Admin';
                }
            } catch (\Exception $exception) {
            }
        }

        ## validate the password.
        return $passwordValidation;
    }

    /**
     * @param Request $request
     * @param TokenInterface $token
     * @param string $providerKey
     *
     * @return RedirectResponse|Response|null
     */
    public
    function onAuthenticationSuccess(
        Request $request,
        TokenInterface $token,
        $providerKey
    ) {

        ## get application settings.
        $this->app_settings->saveAppSettingsInSession();
        $targetPath = $this->getTargetPath($request->getSession(), $providerKey);

        ## if target path is not empty or not equal to the normal user login path
//        if (!empty($targetPath)) {
//            return new RedirectResponse($targetPath);
//        }

        ## generate Url with default locale if exits.
        $defaultLocale = $this->app_settings->getAppSettingsFromSessions('default_language');
        if (empty($defaultLocale)) {
            $defaultLocale = 'en';
        }

        if ($this->stateLessAuth) {
            ## if this is stateless authentication then return the user info and the API token that generated.
//            return $this->generatedToken;
            $response = new Response($this->generatedToken);
            $response->headers->clearCookie('PHPSESSID');
            $response->headers->remove('Set-Cookie');
            $response->send();

            return $response;
        } else {
            ## if this is not stateless authentication then redirect to the login page.
            ## redirect to the homepage
            return new RedirectResponse($this->urlGenerator->generate('HomePage', ['_locale' => $defaultLocale]));
        }
    }

    /**
     * @return bool
     */
    public
    function supportsRememberMe()
    {
        return true;
    }

    /**
     * Return the URL to the login page,if user is not authenticate.
     *
     * @return string
     */
    protected
    function getLoginUrl()
    {

        if (empty($this->request_from_route)) {
            $this->request_from_route = 'user_login';
        }

        return $this->urlGenerator->generate($this->request_from_route);
    }

    /*
     * Set guardian in the security token
     * */
    protected
    function setGuardianInSecurityToken()
    {

    }
}
