<?php

namespace App\Controller;

use App\Entity\CustomFields;
use App\Entity\CustomFieldSections;
use App\Form\AddingCustomFieldsToUserFormType;
use App\Form\Model\AddingCustomFieldsToUserFormModel;
use App\Repository\CustomFieldSectionsRepository;
use App\Repository\CustomFieldsRepository;
use App\Service\DefaultFunction;
use Doctrine\ORM\EntityManagerInterface;
use Exception;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response as ResponseAlias;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @IsGranted("ROLE_USER")
 * IMPROVEMENT: add new field and update exiting field should be handle from the symfony form component.
 * */
class CustomFieldsController extends AbstractController {


	/**
	 * @var CustomFieldsRepository
	 */
	private $custom_fields_repository;
	/**
	 * @var EntityManagerInterface
	 */
	private $entity_manager;
	/**
	 * @var CustomFieldSectionsRepository
	 */
	private $custom_field_sections_repository;
	/**
	 * @var DefaultFunction
	 */
	private $default_function;


	/**
	 * DefaultController constructor.
	 *
	 * @param CustomFieldsRepository        $custom_fields_repository
	 * @param EntityManagerInterface        $entity_manager
	 * @param CustomFieldSectionsRepository $custom_field_sections_repository
	 * @param DefaultFunction               $default_function
	 */
	public function __construct( CustomFieldsRepository $custom_fields_repository, EntityManagerInterface $entity_manager, CustomFieldSectionsRepository $custom_field_sections_repository, DefaultFunction $default_function ) {
		$this->custom_fields_repository         = $custom_fields_repository;
		$this->entity_manager                   = $entity_manager;
		$this->custom_field_sections_repository = $custom_field_sections_repository;
		$this->default_function                 = $default_function;
	}

	/**
	 * @Route("/{_locale}/settings/custom-fields", name="custom_fields")
	 * This method will show list of all custom fields.
	 *
	 * @param Request $request
	 *
	 * @return ResponseAlias
	 */
	public function index( Request $request ) {
		if ( $request->query->get( 'query' ) ) {
			## if we've to search something.
			$CustomFieldList = $this->custom_fields_repository->search( $request->query->get( 'query' ) );
		} else {
			## all field list
			$CustomFieldList = $this->custom_fields_repository->findBy( [], [ 'sorting_order' => 'asc' ] );
		}

		## rendering html.
		return $this->render( 'CustomFields/index.html.twig', [
			'fieldsLIst'                => $CustomFieldList,
			'fieldsSection'             => $this->custom_field_sections_repository->findBy( [], [ 'sorting_order' => 'asc' ] ),
			'noOfFieldInDefaultSection' => $this->custom_fields_repository->findBy( [ 'customFieldSections' => 1 ] ),
		] );
	}

	/**
	 * @Route("{_locale}/settings/add-new-field/{id}", name="add_custom_fields", defaults={"id"=null})
	 * PURPOSE: This will render add new field form or update existing field form
	 *
	 * @param null $id
	 *
	 * @return ResponseAlias
	 */
	public function renderingAddNewFieldForm( $id = null ) {

		try {
			## if client request for the field edition
			$fieldDetails = $this->custom_fields_repository->find( $id );
		} catch ( Exception $exception ) {
			## if table or repository is not found then or any other error
			$fieldDetails = new CustomFields();
		}

		## if user request for that field which is not in database
		if ( ! $fieldDetails instanceof CustomFields ) {
			$fieldDetails = new CustomFields();
		}


		return $this->render( 'CustomFields/addNewField.html.twig', [
			'customFieldForm'     => $this->createForm( AddingCustomFieldsToUserFormType::class )->createView(),
			'fieldDetails'        => $fieldDetails,
			'fieldSections'       => $this->custom_field_sections_repository->findAll(),
			'fieldSectionForEdit' => new CustomFieldSections(),
		] );
	}

	/**
	 * @Route("{_locale}/settings/add-new-section/{id}", name="add_custom_field_section", defaults={"id"=null})
	 * PURPOSE: This will render add new section and update an existing section
	 *
	 * @param null $id
	 *
	 * @return ResponseAlias
	 */
	public function renderingAddNewSectionForm( $id = null ) {

		try {
			## if client request for the section edit
			$fieldSectionForEdit = $this->custom_field_sections_repository->find( $id );
		} catch ( Exception $exception ) {
			## if table or repository is not found then or any other error
			$fieldSectionForEdit = new CustomFieldSections();
		}

		## if user request for that field which is not in database
		if ( ! $fieldSectionForEdit instanceof CustomFieldSections ) {
			$fieldSectionForEdit = new CustomFieldSections();
		}


		return $this->render( 'CustomFields/addNewField.html.twig', [
			'customFieldForm'     => $this->createForm( AddingCustomFieldsToUserFormType::class )->createView(),
			'fieldDetails'        => new CustomFields(),
			'fieldSections'       => $this->custom_field_sections_repository->findAll(),
			'fieldSectionForEdit' => $fieldSectionForEdit,
		] );
	}

	/**
	 * @Route("/settings/save-new-field", name="save_custom_field")
	 * @param Request $request
	 *
	 * @return ResponseAlias
	 *  PURPOSE: Save updated field data to database of add new form field or edit exiting form field.
	 */
	public function submitAddNewFieldForm( Request $request ) {

		## field visible in registration form is dependent on the visible field.
		## we can only accept those value for the visible in registration form of which are in the visible to field.
		$fieldVisibleInTheseUsers = $request->request->get( 'visible_in_registration_form_of', [] );
		$fieldVisibleTo           = $request->request->get( 'visible_to', [] );


		## remove the staff role from the visible to registration form field if its not in the visible field.
		foreach ( $fieldVisibleInTheseUsers as $key_fv => $value_fv ) {
			if ( ! in_array( $value_fv, $fieldVisibleTo ) ) {
				unset( $fieldVisibleInTheseUsers[ $key_fv ] );
			} else {
				## remove white spaces from string.
				$fieldVisibleInTheseUsers[ $key_fv ] = trim( $value_fv );
			}
		}

		## fields options which is selected by select2.
		$fieldOptions = $request->request->get( 'options' );
		## field visible to
		$fieldVisibleTo = implode( ',', $fieldVisibleTo );
		## field show in following user's registration form
		$fieldVisibleInTheseUsers = implode( ',', $fieldVisibleInTheseUsers );

		## allowed extensions.
		$allowedExtensions = implode( ',', $request->request->get( 'allowed_extension', [] ) );
		## id of the field which we want to the update.
		$fieldId = $request->request->get( 'id' );
		## field section
		$fieldSection = $request->request->get( 'section' );

		## message that need to return
		$returnMessage = '';


		## if empty field option then add a simple array.
		if ( empty( $fieldOptions ) || ! is_array( $fieldOptions ) ) {
			$fieldOptions = [];
		}

		/** @var CustomFields $customFieldEntity */
		$customFieldEntity = new CustomFields();

		## creating a form.
		$CustomFieldForm = $this->createForm( AddingCustomFieldsToUserFormType::class );

		## checking form is submitting or not.
		$CustomFieldForm->handleRequest( $request );

		## if form is submitted & validate
		if ( $CustomFieldForm->isSubmitted() && $CustomFieldForm->isValid() ) {


			/** @var AddingCustomFieldsToUserFormModel $CustomFieldFormData */
			$CustomFieldFormData = $CustomFieldForm->getData();
			## separate array values, to make it single dimensional array.
			$CustomFieldFormData->setOptions( array_values( $fieldOptions ) );
			$CustomFieldFormData->setVisibleTo( $fieldVisibleTo );
			$CustomFieldFormData->setVisibleInRegistrationFormOf( $fieldVisibleInTheseUsers );
			$CustomFieldFormData->setAllowedExtension( $allowedExtensions );
			try {


				## get the field section
				$section = $this->custom_field_sections_repository->find( $fieldSection );

				## if section is not found get the default section
				if ( ! $section instanceof CustomFieldSections ) {
					$section = $this->custom_field_sections_repository->find( CustomFieldSectionsRepository::DefaultSectionRef );
				}
				$CustomFieldFormData->setSection( $section );
			} catch ( Exception $exception ) {

			}


			## udpate the field
			if ( $fieldId ) {
				$customFieldEntity = $this->custom_fields_repository->find( $fieldId );
			}

			## client send null is require check is not checked - else if is checked then we get the 'on' keyword
			$is_require = 0;
			if ( $request->request->get( 'is_require' ) == 'on' ) {
				$is_require = 1;
			}

			## setting value in the entity.
			$customFieldEntity->setLabel( $CustomFieldFormData->getLabel() );
			$customFieldEntity->setType( $CustomFieldFormData->getType() );
			$customFieldEntity->setis_require( $is_require );
			$customFieldEntity->setPlaceholder( $CustomFieldFormData->getPlaceholder() );
			$customFieldEntity->setOptions( $CustomFieldFormData->getOptions() );
			$customFieldEntity->setVisibleTo( $CustomFieldFormData->getVisibleTo() );
			$customFieldEntity->setVisibleToRegistrationFormOf( $CustomFieldFormData->getVisibleInRegistrationFormOf() );
			$customFieldEntity->setAllowedExtension( $CustomFieldFormData->getAllowedExtension() );
			$customFieldEntity->setCustomFieldSections( $CustomFieldFormData->getSection() );
			$customFieldEntity->setHelpMessage( $CustomFieldFormData->getHelpMessage() );

			try {

				## save data to database
				$this->entity_manager->persist( $customFieldEntity );
				## remove the previous object.
				$this->entity_manager->flush();

				## set sorting order of the custom field.
				$customFieldEntity->setSortingOrder( $customFieldEntity->getId() );
				## save data to database
				$this->entity_manager->persist( $customFieldEntity );
				## remove the previous object.
				$this->entity_manager->flush();

				$returnMessage = 'OK';
			} catch ( Exception $exception ) {
				$returnMessage = $exception->getMessage();
			}


			## if we are adding or updating new field.
			return new JsonResponse( $returnMessage, 200 );
		}

		return $this->render( 'CustomFields/ajaxPatches/addNewField.html.twig', [
			'customFieldForm' => $CustomFieldForm->createView(),
			'fieldDetails'    => $customFieldEntity,
			'returnMessage'   => $returnMessage,
			'fieldSections'   => $this->custom_field_sections_repository->findAll(),
		] );
	}

	/**
	 * @Route("/ajax/request/", name="ajax_request_customField")
	 * @param Request $request
	 *
	 * @return JsonResponse
	 */
	public function ajaxHandler( Request $request ) {
		$returnValue = '';

		switch ( $request->query->get( 'query' ) ) {
			case 'delete_field':
				// delete field if exits.
				try {
					$response = $this->custom_fields_repository->find( $request->request->get( 'target' ) );
					$this->entity_manager->remove( $response );
					$this->entity_manager->flush();
					$returnValue = 'OK';
				} catch ( Exception $e ) {
					$returnValue = $e->getMessage();
				}
				break;
			case 'save_section':
				// save new section
				$response = [];
				if ( empty( $request->request->get( 'name' ) ) ) {
					$returnValue = $this->default_function->push_error( $response, 'Please type section name' );

				} else {

					/** @var CustomFields $customFieldEntity */
					if ( empty( $request->request->get( 'id' ) ) ) {
						$customFieldSectionEntity = new CustomFieldSections();
					} else {
						$customFieldSectionEntity = $this->custom_field_sections_repository->find( $request->request->get( 'id' ) );
					}

					## set the name.
					$customFieldSectionEntity->setName( $request->request->get( 'name' ) );
					$customFieldSectionEntity->setDescription( $request->request->get( 'description' ) );
					$this->entity_manager->persist( $customFieldSectionEntity );
					$this->entity_manager->flush();

					## set sorting order of the field.
					$customFieldSectionEntity->setSortingOrder( $customFieldSectionEntity->getId() );
					$this->entity_manager->persist( $customFieldSectionEntity );
					$this->entity_manager->flush();


					$returnValue = 'OK';
				}
				break;
			case 'sorting_field':
				// sorting custom field
				$custom_field = $request->request->get( 'fieldList_tr' );
				if ( is_array( $custom_field ) && ! empty( $custom_field ) ) {
					for ( $x = 0; $x < count( $custom_field ); $x ++ ) {
						$CustomFieldSection = $this->custom_fields_repository->find( $custom_field[ $x ] );
						$CustomFieldSection->setSortingOrder( (int) ( $x ) + 1 );
						$this->entity_manager->persist( $CustomFieldSection );
					}
					$this->entity_manager->flush();
				}
				break;
			case 'sort_sections':
				// sorting sections of custom field
				$section_orders = $request->request->get( 'f_section' );
				if ( is_array( $section_orders ) && ! empty( $section_orders ) ) {
					for ( $x = 0; $x < count( $section_orders ); $x ++ ) {
						$CustomFieldSection = $this->custom_field_sections_repository->find( $section_orders[ $x ] );
						$CustomFieldSection->setSortingOrder( (int) ( $x ) + 1 );
						$this->entity_manager->persist( $CustomFieldSection );
					}
					$this->entity_manager->flush();
				}
				break;
			case 'delete_section':
				// delete section
				// if selected section have fields then these fields will move to default section.
				// default custom field section will be at the first.
				try {
					## get the custom field
					$section_ = $this->custom_field_sections_repository->find( $request->request->get( 'target' ) );
					$response = $section_->getCustomFieldRef()->toArray();

					## get the custom field against the section of custom field
					/** @var CustomFields $value */
					foreach ( $response as $key => $value ) {
						$value->setCustomFieldSections( $this->custom_field_sections_repository->find( 1 ) );
						$this->entity_manager->persist( $value );
					}

					## delete the section of custom field
					$this->entity_manager->remove( $section_ );
					## make changes in the database.
					$this->entity_manager->flush();

					## return success response.
					$returnValue = 'OK';
				} catch ( Exception $e ) {
					$returnValue = $e->getMessage();
				}

				break;

			default:
				$returnValue = 'No route found';
				break;

		}


		/*		if ( $request->query->get( 'query' ) == 'delete_field' ) {
					try {
						$response = $this->custom_fields_repository->find( $request->request->get( 'target' ) );
						$this->entity_manager->remove( $response );
						$this->entity_manager->flush();
						$returnValue = 'OK';
					} catch ( Exception $e ) {
						$returnValue = $e->getMessage();
					}
				} else if ( $request->query->get( 'query' ) == 'save_section' ) {
					$response = [];
					if ( empty( $request->request->get( 'name' ) ) ) {
						$returnValue = $this->default_function->push_error( $response, 'Please type section name' );

					} else {
						if ( empty( $request->request->get( 'id' ) ) ) {
							$customFieldSectionEntity = new CustomFieldSections();
						} else {
							$customFieldSectionEntity = $this->custom_field_sections_repository->find( $request->request->get( 'id' ) );
						}

						## set the name.
						$customFieldSectionEntity->setName( $request->request->get( 'name' ) );
						$customFieldSectionEntity->setDescription( $request->request->get( 'description' ) );
						$this->entity_manager->persist( $customFieldSectionEntity );
						$this->entity_manager->flush();
						$returnValue = 'OK';
					}

				} else if ( $request->query->get( 'query' ) == 'sort_sections' ) {
					// sorting sections of custom field
					$section_orders = $request->request->get( 'f_section' );
					if ( is_array( $section_orders ) && ! empty( $section_orders ) ) {
						for ( $x = 0; $x < count( $section_orders ); $x ++ ) {
							$CustomFieldSection = $this->custom_field_sections_repository->find( $section_orders[ $x ] );
							$CustomFieldSection->setSortingOrder( (int) ( $x ) + 1 );
							$this->entity_manager->persist( $CustomFieldSection );
						}
						$this->entity_manager->flush();
					}
				} else if ( $request->query->get( 'query' ) == 'sorting_field' ) {
					// sorting custom field
					$custom_field = $request->request->get( 'fieldList_tr' );
					if ( is_array( $custom_field ) && ! empty( $custom_field ) ) {
						for ( $x = 0; $x < count( $custom_field ); $x ++ ) {
							$CustomFieldSection = $this->custom_fields_repository->find( $custom_field[ $x ] );
							$CustomFieldSection->setSortingOrder( (int) ( $x ) + 1 );
							$this->entity_manager->persist( $CustomFieldSection );
						}
						$this->entity_manager->flush();
					}
				} else if ( $request->query->get( 'delete_section' ) ) {
					dump( $request->request );
				}*/

		return new JsonResponse( $returnValue );
	}


}
