<?php

namespace App\Controller;

use App\Entity\AppSettings;
use App\Entity\Attendance;
use App\Entity\AttendanceStatus;

use App\Entity\Guardian;
use App\Repository\AllocateTeacherToClassAndSubjectsRepository;
use App\Repository\AttendanceRepository;
use App\Repository\AttendanceStatusRepository;
use App\Repository\ClassesRepository;
use App\Repository\ClassSessionEnrolmentRepository;
use App\Repository\UserRepository;
use App\Service\Attendance\Base;
use App\Service\Constants;
use App\Service\DefaultFunction;
use App\Service\Guardian\GuardianService;
use App\Service\PrintService\PrintService;
use App\Service\StudentClasses;
use App\Service\StudentsEnrollment\Enrollments;
use App\Service\SubjectService;
use http\Client\Response;
use Knp\Bundle\SnappyBundle\Snappy\Response\PdfResponse;
use Knp\Snappy\Pdf;
use Nzo\UrlEncryptorBundle\Annotations\ParamDecryptor;
use Nzo\UrlEncryptorBundle\UrlEncryptor\UrlEncryptor;
use phpDocumentor\Reflection\DocBlock\Tags\Deprecated;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bridge\Twig\Tests\Extension\Subject;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response as ResponseAlias;
use Symfony\Component\Routing\Annotation\Route;

/**
 *
 * @IsGranted("ROLE_USER")
 * */
class AttendanceController extends AbstractController
{
    /**
     * @var Base
     */
    private $attendance_baseService;
    /**
     * @var AttendanceStatusRepository
     */
    private $attendance_status_repository;
    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var ClassesRepository
     */
    private $classes_repository;
    /**
     * @var ClassSessionEnrolmentRepository
     */
    private $class_session_enrolment_repository;
    /**
     * @var Enrollments
     */
    private $enrollments;
    /**
     * @var AttendanceRepository
     */
    private $attendance_repository;
    /**
     * @var UrlEncryptor
     */
    private $url_encryptor;
    /**
     * @var Pdf
     */
    private $pdf;
    /**
     * @var DefaultFunction
     */
    private $defaultFunction;
    /**
     * @var AppSettings
     */
    private $appSettings;
    /**
     * @var AllocateTeacherToClassAndSubjectsRepository
     */
    private $allocateTeacherToClassAndSubjectsRepository;

    public function __construct(
        Base $attendance_baseService,
        AttendanceStatusRepository $attendance_status_repository,
        UserRepository $user_repository,
        ClassesRepository $classes_repository,
        ClassSessionEnrolmentRepository $class_session_enrolment_repository,
        Enrollments $enrollments,
        AttendanceRepository $attendance_repository,
        UrlEncryptor $url_encryptor,
        Pdf $pdf,
        DefaultFunction $defaultFunction,
        \App\Service\AppSettings $appSettings,
        AllocateTeacherToClassAndSubjectsRepository $allocateTeacherToClassAndSubjectsRepository
    ) {
        $this->attendance_baseService = $attendance_baseService;
        $this->attendance_status_repository = $attendance_status_repository;
        $this->user_repository = $user_repository;
        $this->classes_repository = $classes_repository;
        $this->class_session_enrolment_repository = $class_session_enrolment_repository;
        $this->enrollments = $enrollments;
        $this->attendance_repository = $attendance_repository;
        $this->url_encryptor = $url_encryptor;
        $this->pdf = $pdf;
        $this->defaultFunction = $defaultFunction;
        $this->appSettings = $appSettings;
        $this->allocateTeacherToClassAndSubjectsRepository = $allocateTeacherToClassAndSubjectsRepository;
    }


    /**
     * @Route("{_locale}/attendance/mark-attendance", name="add_attendance")
     */
    public function addAttendance()
    {
        return $this->__user_list_on_user_base([], 'addAttendance', 'render', 0, true);
    }

    /**
     * @IsGranted("ROLE_ADMIN")
     * @Route("{_locale}/attendance/add/status/{query}", name="add_attendance_status", defaults={"query"=null})
     * @ParamDecryptor(params={"query"})
     */
    public function addAttendanceStatus(
        $query
    ) {


        ## get the attendance record
        $AttendanceStatus = $this->attendance_baseService->getAttendanceStatus($query);

        ## if attendance is not found against the query then return the empty object
        if (!$AttendanceStatus instanceof AttendanceStatus) {
            $AttendanceStatus = new AttendanceStatus();
        }

        return $this->render(
            'attendance/addStatus.html.twig',
            [
                'attendance_status__obj' => $AttendanceStatus,
            ]
        );
    }

    /**
     * @Route("{_locale}/attendance/status-list/", name="attendance_status")
     * @IsGranted("ROLE_ATTENDANCE_MANAGER")
     */
    public function attendanceStatus()
    {
        return $this->render(
            'attendance/AttendanceStatus.html.twig',
            [
                'attendance_status___list' => $this->attendance_status_repository->findAll(),
            ]
        );
    }

    ## get the user list on based of user
    private function __user_list_on_user_base(
        $UserList,
        $template = 'addAttendance',
        $render = 'render',
        $detail_check = false,
        $donot_showDetailsOfAttendance = false,
        $in_date = null
    ) {


        $passVariable = [
            ## user list
            '__user_list' => $UserList,
            ## get classes of active sessions
            'activeClassesWithSession' => $this->classes_repository->getActiveClassesOfActiveSession(),
            'deep_view' => $detail_check,
            ## get the enable status of attendance
            'attendance__status' => $this->attendance_status_repository->findBy(['enable' => 1]),
            ## don't show list of the attendance.
            'donot_show_list_of_attendance' => $donot_showDetailsOfAttendance,
            'in_date' => $in_date,


            /**
             * @deprecated - 01-October-2019
             * */ ## get the active class of active sessions
            'activeClassesWithSession__not_subject_wise' => $this->classes_repository->getClassWiseAttendanceClassesList(
            ),
            'activeClassesWithSession__subject_wise' => $this->classes_repository->getSubjectWiseAttendanceClassesList(
            ),
        ];

        if ($this->getUser()->getCategory() === 't') {
            $passVariable['assignedClassesToTeacher'] = $this->allocateTeacherToClassAndSubjectsRepository->getUniqueClassAllocationByTeacherID(
                $this->getUser()->getId()
            );
        }

        return $this->$render(
            'attendance/'.$template.'.html.twig',
            $passVariable
        );
    }

    ## load attendances of student
    private function load___addAttendances(ParameterBag $bag)
    {
        return $this->__user_list_on_user_base(
            $this->class_session_enrolment_repository->load____attendances($bag),
            $bag->get('template', 'addAttendance'),
            $bag->get('render', 'render'),
            $bag->get('deep_level', 3),
            false,
            $bag->get('in_date')
        );
    }

    /**
     * @Route("{_locale}/attendance/requestHandler/{query}", name="requestHandler__attendance")
     * @param Request $request
     * @param null $query
     *
     * @return JsonResponse
     */
    public function ajaxHandler(Request $request, $query = null)
    {
        ## if user is not logged in the redirect th e use rto login page.
        $response = 'OK';

        if (empty($query)) {
            ## get the query variable from the get request.
            $query = $request->query->get('query');
        }

        ## for direct the ajax request for the specific action
        switch ($query) {
            case 'add___attendanceStatus':
                ## add attendance status.
                $response = $this->attendance_baseService->saveAttendanceStatus($request->request);
                break;
            case 'delete__attendanceStatus':
                ## load user on based of class enrollment.
                $response = $this->attendance_baseService->deleteAttendanceStatus(
                    $this->url_encryptor->decrypt($request->request->get('ref'))
                );
                break;
            case 'add___attendance':
                ## add attendance of users
                $response = $this->attendance_baseService->bulkSaveAttendance($request->request);
                //                if ($response instanceof Attendance) {
                //
                //                    ## adding some extra parameters.
                //                    $response_['user_id_id'] = $request->request->get('id');
                //                    $response_['attendance_ref'] = $response->getId();
                //                    $response_['attendance_status_type'] = $response->getAttendanceStatusType();
                //                    $response_['attendance_status_name'] = $response->getAttendanceStatus()->getName();
                //                    $response_['attendance_status_id'] = $request->get('__attendance__status');
                //                    $response_['datetime_in'] = $response->getDatetimeIn();
                //                    $response_['datetime_out'] = $response->getDatetimeOut();
                //                    $response_['comments'] = $response->getComments();
                //                    $response_['fName'] = $response->getUser()->getFName();
                //                    $response_['lname'] = $response->getUser()->getLName();
                //                    $response = [$response_];
                //
                //                    $response = 'escape__fromErro_' . $this->__user_list_on_user_base($response, '___list__addAttendance', 'renderView', 3);
                break;
            case 'load___userFilterBase':
                ## load user on based of class enrollment.
                $response = 'escape__fromErro_'.$this->__user_list_on_user_base(
                        $this->enrollments->getListOfUserBasedOnClassEnrollments($request->request),
                        '___list__addAttendance',
                        'renderView',
                        1
                    );
                break;
            case 'load___userOnSubjectBase':
                ## load user on based of class enrollment.
                $response = 'escape__fromErro_'.$this->__user_list_on_user_base(
                        $this->enrollments->getEnrolledUserOnBaseOfSubjects($request->request),
                        '___list__addAttendance',
                        'renderView',
                        2
                    );
                break;
            case 'load___attendance':
                if ($request->request->get('ld') == 'st') {
                    ## load staff or teacher attendance
                    $bag = new ParameterBag();
                    $bag->set('in_date', $request->get('in_date'));
                    $bag->set('userCatRef', $request->get('userCategory', 't'));
                    $response = 'escape__fromErro_'.$this->renderView(
                            'attendance/___list__addAttendanceOfTeacherAndStaff.html.twig',
                            [
                                '__user_list' => $this->user_repository->loadAttendanceOfTeacherAndStaff($bag),
                                'attendance__status' => $this->attendance_status_repository->findBy(['enable' => 1]),
                                'in_date' => $bag->get('in_date'),
                                'userCategory' => $bag->get('userCategory'),
                            ]
                        );
                } else {
                    ## load student attendance
                    if (empty($request->request->get('active__classes'))) {
                        $response = 'Class not selected';
                    } else {
                        ## load attendances based on the filters.
                        $request->request->set('template', '___list__addAttendance');
                        $request->request->set('render', 'renderView');
                        $request->request->set('deep_level', 3);

                        $response = 'escape__fromErro_'.$this->load___addAttendances($request->request);
                    }
                }


                break;
        }

        return new JsonResponse($response, 200);
    }


    /**
     * ## Reporting Controller
     *
     * */

    /**
     * @Route("{_locale}/attendance", name="attendance")
     *
     * @param Request $request
     *
     * @return ResponseAlias
     */
    public function index(Request $request)
    {
        return $this->render(
            'attendance/attendanceList.html.twig',
            [
                ## Active class with their session reference
                'activeClassesWithSession' => $this->classes_repository->getClassesOfActiveSession(),
                ## not of present persons in a class
                'noOfPresentStudentInClass' => $this->classes_repository->getNoOfPresentStudentInActiveClass(
                    $request->query->get('date')
                ),
            ]
        );
    }

    /**
     * @Route("{_locale}/attendance/daliy-attendance/", name="daily_attendance")
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     * @IsGranted("ROLE_ATTENDANCE_MANAGER")
     * @param StudentClasses $studentClasses
     *
     * @return ResponseAlias
     * @ParamDecryptor(params={"class"})
     *  PURPOSE: Used for showing attendance of students class wise.
     */
    public function DailyAttendance(Request $request, DefaultFunction $defaultFunction, StudentClasses $studentClasses)
    {
        if (!empty($request->get('class'))) {
            $request->query->set('class', $this->url_encryptor->decrypt($request->query->get('class')));
        }

        $attendances = null;
        if (count($request->query)) {
            $attendances = $this->attendance_repository->reporting__attendances($request->query);
        }

        ## variable data object
        $sendingData = [
            'attendances' => $attendances,
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {
            $filename = 'Class based Attendance report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('DailyReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Daily.Report/default.html.twig';
            }

            ##
            $sendingData['class'] = null;
            if (!empty($request->query->get('class'))) {
                $sendingData['class'] = $this->classes_repository->find($request->query->get('class'));
            }

            $sendingData['attendance_status'] = [];
            if (!empty($request->query->get('attendance__status'))) {
                $sendingData['attendance_status'] = $this->attendance_status_repository->find(
                    $request->query->get('attendance__status')
                );
            }

            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        $sendingData['classes'] = $studentClasses->getAllClasses($request->query);
        $sendingData['attendanceStatus__list'] = $this->attendance_status_repository->findAll();

        return $this->render('attendance/Reporting/dailyAttendance.html.twig', $sendingData);
    }


    /**
     * @Route("{_locale}/guardian/attendance/daliy-attendance/", name="daily_attendance_guardian")
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     * @IsGranted("ROLE_GUARDIAN")
     * @param StudentClasses $studentClasses
     *
     * @return ResponseAlias
     *  PURPOSE: Show daily attendance of connected student, childrens to guardian.
     */
    public function DailyAttendanceGuardian()
    {
        $bag = new ParameterBag();
        $bag->set('userArray', GuardianService::getAttachedStudentsIdWithGuardian($this->getUser()));
        $bag->set('attend_inTime', date('Y-m-d'));

        return $this->render(
            'attendance/Reporting/guardian/dailyAttendance.html.twig',
            [
                'attendances' => $this->attendance_repository->reporting__attendances($bag),
            ]
        );
    }


    /**
     * @Route("{_locale}/guardian/attendance/filter-student-attendance/", name="filter_my_children_attendance")
     * @param Request $request
     *
     * @param DefaultFunction $defaultFunction
     * @param StudentClasses $studentClasses
     * @param SubjectService $subjectService
     * @IsGranted("ROLE_GUARDIAN")
     * @return ResponseAlias
     * Purpose: Filter Attendance
     */
    public function filterAttendanceMyChildren(
        Request $request,
        DefaultFunction $defaultFunction,
        StudentClasses $studentClasses,
        SubjectService $subjectService
    ) {

        $request->query->set('userCategory', 's');
        $attendanceList = null;
        if (count($request->query) > 0) {
            $attendanceList = $this->attendance_repository->reporting__attendances($request->query);
        }

        $sendingData = [
            'attendance' => $attendanceList,
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {

            $filename = 'Attendance Filter Report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('FilterAttendanceReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Filter.Attendance.Report/default.html.twig';
            }


            $sendingData['user'] = $this->user_repository->find($request->query->get('student__based'));
            $sendingData['class'] = $this->classes_repository->find($request->query->get('class'));
            $sendingData['classSubject'] = $subjectService->get__ClassSubject(
                (int)$request->query->get('classSubject')
            );
            $sendingData['attendance__status'] = $this->attendance_status_repository->find(
                $request->query->get('attendance__status')
            );
            $sendingData['attendance__markBy'] = $this->user_repository->find(
                $request->query->get('attendance__markBy')
            );

            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        ## student list

        $sendingData['student_list'] = $this->user_repository->findAll();
        $sendingData['user_list'] = $this->user_repository->findAll();


        $sendingData['classes_list'] = $studentClasses->getAllClasses();
        $sendingData['classesSubject_list'] = $subjectService->listOfClassSubjects();
        $sendingData['attendanceStatus__list'] = $this->attendance_status_repository->findAll();

        return $this->render('attendance/Reporting/guardian/__filterAttendanceReportBase.html.twig', $sendingData);
    }


    /**
     * @Route("{_locale}/student/attendance/filter-student-attendance/", name="filter_my_attendance")
     * @param Request $request
     *
     * @param DefaultFunction $defaultFunction
     * @param StudentClasses $studentClasses
     * @param SubjectService $subjectService
     * @IsGranted("ROLE_STUDENT")
     * @return ResponseAlias
     * Purpose: Filter Attendance
     */
    public function filterAttendanceStudent(
        Request $request,
        DefaultFunction $defaultFunction,
        StudentClasses $studentClasses,
        SubjectService $subjectService
    ) {

        $attendanceList = null;
        if (count($request->query) > 0) {
            $attendanceList = $this->attendance_repository->reporting__attendances($request->query);
        }

        $sendingData = [
            'attendance' => $attendanceList,
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {

            $filename = 'Attendance Filter Report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('FilterAttendanceReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Filter.Attendance.Report/default.html.twig';
            }


            $sendingData['user'] = $this->user_repository->find($request->query->get('student__based'));
            $sendingData['class'] = $this->classes_repository->find($request->query->get('class'));
            $sendingData['classSubject'] = $subjectService->get__ClassSubject(
                (int)$request->query->get('classSubject')
            );
            $sendingData['attendance__status'] = $this->attendance_status_repository->find(
                $request->query->get('attendance__status')
            );
            $sendingData['attendance__markBy'] = $this->user_repository->find(
                $request->query->get('attendance__markBy')
            );

            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        ## student list

        $sendingData['student_list'] = $this->user_repository->findAll();
        $sendingData['user_list'] = $this->user_repository->findAll();
        $sendingData['classes_list'] = $studentClasses->getAllClasses();
        $sendingData['classesSubject_list'] = $subjectService->listOfClassSubjects();
        $sendingData['attendanceStatus__list'] = $this->attendance_status_repository->findAll();

        return $this->render('attendance/Reporting/student/__filterAttendanceReportBase.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/attendance/class", name="class_wise_attendance",
     *                               defaults={"class"=null,"className"=null})
     * @param                 $className
     * @param Request $request
     * @param DefaultFunction $defaultFunction
     * @param null $class
     *
     * @param StudentClasses $studentClasses
     *
     * @return ResponseAlias
     * @ParamDecryptor(params={"class"})
     *  PURPOSE: Used for showing attendance of students class wise.
     */
    public function classBased(
        $className,
        Request $request,
        DefaultFunction $defaultFunction,
        StudentClasses $studentClasses,
        $class = null
    ) {

        $bag = new ParameterBag();
        $bag->set('attendance_markingDate', $request->query->get('date'));

        ## variable data object
        $sendingData = [
            'dateRange' => $request->query->get('date'),
        ];

        ## if class is not empty then search
        if (!empty($request->query->get('class'))) {
            $sendingData['attendanceReportOfAClass'] = $this->class_session_enrolment_repository->getAttendanceOfClass(
                (int)$this->url_encryptor->decrypt($request->query->get('class')),
                $this->attendance_repository,
                $bag
            );

            $filename = 'Class based Attendance report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('StudentbaseAttendanceReportDaywise')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/student.root/Student.base.Attendance.Report.Day.wise/default.html.twig';
            }
            $sendingData['class'] = $this->classes_repository->find(
                $this->url_encryptor->decrypt($request->query->get('Class'))
            );
            $html_fe = $this->renderView($reportFile, $sendingData);


            return new ResponseAlias(
                $this->pdf->getOutputFromHtml(
                    $html_fe,
                    [
                        'orientation' => 'Landscape',
                    ]
                ), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        $sendingData['classes'] = $studentClasses->getAllClasses();

        return $this->render('attendance/Reporting/_class_based_detail_attendance.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/attendance/student/{id}/{Uname}", name="student_wise_attendance", defaults={"id"=null,
     *                                                     "Uname"=null})
     * @param Request $request
     * @param null $id
     * @param null $Uname
     *
     * @param DefaultFunction $defaultFunction
     *
     * @return ResponseAlias
     * @ParamDecryptor(params={"id"})
     *  PURPOSE: Used for showing attendance of students
     */
    public function studentBased(Request $request, DefaultFunction $defaultFunction, $id = null, $Uname = null)
    {
        $Bag = new ParameterBag();
        ## student based.
        $Bag->set('student__based', $id);
        ## search attendance wise duration.
        $Bag->set('attendance_markingDate', $request->query->get('date'));

        $sendingData = [
            'attendanceReportOfAClass' => $this->attendance_repository->reporting__attendances($Bag),
            'uName' => $Uname,
        ];


        ## print pdf report
        if ($request->query->get('printResult') == 1) {
            $filename = 'Student Attendance Evaluation report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('StudentBaseAttendanceReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Student.Base.Attendance.Report/default.html.twig';
            }
            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        return $this->render('attendance/Reporting/_student_based_detail_attendance.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/attendance/class-by-attendance-status/", name="status_wise_attendance")
     * @param Request $request
     *
     * @param DefaultFunction $defaultFunction
     *
     * @return ResponseAlias
     *  PURPOSE: Used for showing attendance of students by status. user select status and this page will serve all
     *  student how are absent or present.
     */
    public function statusBased(Request $request, DefaultFunction $defaultFunction)
    {
        $Bag = new ParameterBag();
        ## search by attendance status.
        $Bag->set('attendance__status', $this->url_encryptor->decrypt($request->query->get('status')));
        ## search by class also
        $Bag->set('class', $this->url_encryptor->decrypt($request->query->get('class')));
        ## attendance range
        $Bag->set('attendance_markingDate', $request->query->get('date'));


        $sendingData = [
            'attendanceReportOfAClass' => $this->attendance_repository->reporting__attendances($Bag),
            'attendanceStatus' => $this->attendance_status_repository->findAll(),
            ## Active class with their session reference
            'activeClassesWithSession' => $this->classes_repository->getClassesOfActiveSession(),
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {
            $filename = 'Class attendance on base of Attendance Status';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('ClassBaseAttendanceStatusAttendanceReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Class.Base.Attendance.Status.Attendance.Report/default.html.twig';
            }
            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        return $this->render('attendance/Reporting/_AttenanceStatus_based_detail_attendance.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/attendance/filter-student-attendance/", name="filter_attendance")
     * @param Request $request
     *
     * @param DefaultFunction $defaultFunction
     * @param StudentClasses $studentClasses
     * @param SubjectService $subjectService
     * @IsGranted("ROLE_ATTENDANCE_MANAGER")
     * @return ResponseAlias
     * Purpose: Filter Attendance
     */
    public function filterAttendance(
        Request $request,
        DefaultFunction $defaultFunction,
        StudentClasses $studentClasses,
        SubjectService $subjectService
    ) {

        $attendanceList = null;
        if (count($request->query) > 0) {
            $attendanceList = $this->attendance_repository->reporting__attendances($request->query);
        }

        $sendingData = [
            'attendance' => $attendanceList,
        ];

        ## print pdf report
        if ($request->query->get('printResult') == 1) {

            $filename = 'Attendance Filter Report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('FilterAttendanceReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Filter.Attendance.Report/default.html.twig';
            }

            $sendingData['user'] = null;
            $sendingData['class'] = null;
            $sendingData['attendance__status'] = null;
            $sendingData['attendance__markBy'] = null;
            if ($request->query->get('student__based')) {
                $sendingData['user'] = $this->user_repository->find($request->query->get('student__based'));
            }
            if ($request->query->get('class')) {
                $sendingData['class'] = $this->classes_repository->find($request->query->get('class'));
            }
            $sendingData['classSubject'] = $subjectService->get__ClassSubject(
                (int)$request->query->get('classSubject')
            );
            if ($request->query->get('attendance__status')) {
                $sendingData['attendance__status'] = $this->attendance_status_repository->find(
                    $request->query->get('attendance__status')
                );
            }
            if ($request->query->get('attendance__markBy')) {
                $sendingData['attendance__markBy'] = $this->user_repository->find(
                    $request->query->get('attendance__markBy')
                );
            }

            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }

        ## student list

        $sendingData['student_list'] = $this->user_repository->findAll();
        $sendingData['user_list'] = $this->user_repository->findAll();


        $sendingData['classes_list'] = $studentClasses->getAllClasses();
        $sendingData['classesSubject_list'] = $subjectService->listOfClassSubjects();
        $sendingData['attendanceStatus__list'] = $this->attendance_status_repository->findAll();

        return $this->render('attendance/Reporting/__filterAttendanceReportBase.html.twig', $sendingData);
    }


    /**
     * @Route("{_locale}/attendance/filter-teacher-attendance/", name="filter_teacher_attendance")
     * @param Request $request
     *
     * @param DefaultFunction $defaultFunction
     * @param StudentClasses $studentClasses
     * @param SubjectService $subjectService
     * @IsGranted("ROLE_TEACHER_MANAGER")
     * @return ResponseAlias
     * Purpose: Filter Attendance
     */
    public function filterTeacherAttendance(
        Request $request,
        DefaultFunction $defaultFunction,
        StudentClasses $studentClasses,
        SubjectService $subjectService
    ) {

        $request->query->set('userCategory', 't');

        $attendanceList = null;
        if (count($request->query) > 0) {
            $attendanceList = $this->attendance_repository->reporting__attendances($request->query);
        }
        $sendingData = [
            'attendance' => $attendanceList,
        ];
        ## print pdf report
        if ($request->query->get('printResult') == 1) {

            $filename = 'Attendance Filter Report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('FilterAttendanceReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Filter.Attendance.Report/default.html.twig';
            }


            $sendingData['user'] = null;
            $sendingData['class'] = null;
            $sendingData['attendance__status'] = null;
            $sendingData['attendance__markBy'] = null;
            if ($request->query->get('student__based')) {
                $sendingData['user'] = $this->user_repository->find($request->query->get('student__based'));
            }
            if ($request->query->get('class')) {
                $sendingData['class'] = $this->classes_repository->find($request->query->get('class'));
            }
            $sendingData['classSubject'] = $subjectService->get__ClassSubject(
                (int)$request->query->get('classSubject')
            );
            if ($request->query->get('attendance__status')) {
                $sendingData['attendance__status'] = $this->attendance_status_repository->find(
                    $request->query->get('attendance__status')
                );
            }
            if ($request->query->get('attendance__markBy')) {
                $sendingData['attendance__markBy'] = $this->user_repository->find(
                    $request->query->get('attendance__markBy')
                );
            }


            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }
        $sendingData['student_list'] = $this->user_repository->findBy(
            [
                'category' => 't',
            ]
        );
        $sendingData['user_list'] = $this->user_repository->findAll();
        $sendingData['attendanceStatus__list'] = $this->attendance_status_repository->findAll();

        return $this->render('attendance/Reporting/__filterTeacherAttendanceReportBase.html.twig', $sendingData);
    }

    /**
     * @Route("{_locale}/attendance/filter-staff-attendance/", name="filter_staff_attendance")
     * @param Request $request
     *
     * @param DefaultFunction $defaultFunction
     *
     * @return ResponseAlias
     * Purpose: Filter Attendance
     */
    public function filterStaffAttendance(
        Request $request,
        DefaultFunction $defaultFunction
    ) {

        $attendanceList = null;
        $request->query->set('userCategory', 'a');

        if (count($request->query) > 0) {
            $attendanceList = $this->attendance_repository->reporting__attendances($request->query);
        }
        $sendingData = [
            'attendance' => $attendanceList,
        ];
        ## print pdf report
        if ($request->query->get('printResult') == 1) {

            $filename = 'Attendance Filter Report';
            $reportFile = $defaultFunction->parseTemplatePathFromFullPath(
                $this->appSettings->getAppSettingsFromSessions('FilterAttendanceReport')
            );
            if (empty($reportFile)) {
                $reportFile = 'pdfPrints/attendance.root/Filter.Attendance.Report/default.html.twig';
            }


            $sendingData['user'] = null;
            $sendingData['class'] = null;
            $sendingData['attendance__status'] = null;
            $sendingData['attendance__markBy'] = null;
            if ($request->query->get('student__based')) {
                $sendingData['user'] = $this->user_repository->find($request->query->get('student__based'));
            }
            if ($request->query->get('class')) {
                $sendingData['class'] = $this->classes_repository->find($request->query->get('class'));
            }
            if ($request->query->get('attendance__status')) {
                $sendingData['attendance__status'] = $this->attendance_status_repository->find(
                    $request->query->get('attendance__status')
                );
            }
            if ($request->query->get('attendance__markBy')) {
                $sendingData['attendance__markBy'] = $this->user_repository->find(
                    $request->query->get('attendance__markBy')
                );
            }

            $html_fe = $this->renderView($reportFile, $sendingData);

            return new ResponseAlias(
                $this->pdf->getOutputFromHtml($html_fe), 200, [
                    'Content-Type' => 'application/pdf',
                    'Content-Disposition' => 'inline; filename="'.$filename.'.pdf"',
                ]
            );
        }
        $sendingData['student_list'] = $this->user_repository->findBy(
            [
                'category' => 'a',
            ]
        );
        $sendingData['user_list'] = $this->user_repository->findAll();
        $sendingData['attendanceStatus__list'] = $this->attendance_status_repository->findAll();

        return $this->render('attendance/Reporting/__filterStaffAttendanceReportBase.html.twig', $sendingData);
    }
}
