<?php


namespace App\Service\accounts;


use App\Entity\ClassFeeSchedule;
use App\Entity\InvoiceItems;
use App\Entity\Invoices;
use App\Entity\Options;
use App\Entity\PaymentMethods;
use App\Entity\Transactions;
use App\Entity\User;
use App\Repository\ClassFeeScheduleRepository;
use App\Repository\InvoiceItemsRepository;
use App\Repository\InvoicesRepository;
use App\Repository\OptionsRepository;
use App\Repository\PaymentMethodsRepository;
use App\Repository\TransactionsRepository;
use App\Repository\UserRepository;
use App\Service\AppSettings;
use App\Service\DefaultFunction;
use App\Service\Fees\FeeSchedulerService;
use App\Service\Fees\StudentFeeService;
use App\Service\SystemServices\systemOptionsService;
use Doctrine\DBAL\Exception\DatabaseObjectNotFoundException;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityNotFoundException;
use Symfony\Component\HttpFoundation\ParameterBag;

class InvoiceService
{

    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var TransactionsRepository
     */
    private $transactions_repository;
    /**
     * @var InvoicesRepository
     */
    private $invoices_repository;
    /**
     * @var InvoiceItemsRepository
     */
    private $invoice_items_repository;
    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var PaymentMethodsRepository
     */
    private $payment_methods_repository;
    /**
     * @var \Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface
     */
    private $token_storage;
    /**
     * @var AccountsService
     */
    private $accounts_service;
    /**
     * @var EntityManagerInterface
     */
    private $entity_manager;
    /**
     * @var AppSettings
     */
    private $app_settings;
    /**
     * @var systemOptionsService
     */
    private $system_options_service;
    /**
     * @var FeeSchedulerService
     */
    private $fee_scheduler_service;
    /**
     * @var ClassFeeScheduleRepository
     */
    private $class_fee_schedule_repository;
    /**
     * @var StudentFeeService
     */
    private $student_fee_service;

    public function __construct(
        StudentFeeService $student_fee_service,
        ClassFeeScheduleRepository $class_fee_schedule_repository,
        FeeSchedulerService $fee_scheduler_service,
        systemOptionsService $system_options_service,
        AppSettings $app_settings,
        DefaultFunction $default_function,
        EntityManagerInterface $entity_manager,
        AccountsService $accounts_service,
        \Symfony\Component\Security\Core\Authentication\Token\Storage\TokenStorageInterface $token_storage,
        UserRepository $user_repository,
        PaymentMethodsRepository $payment_methods_repository,
        InvoicesRepository $invoices_repository,
        InvoiceItemsRepository $invoice_items_repository
    ) {
        $this->default_function = $default_function;
        $this->invoices_repository = $invoices_repository;
        $this->invoice_items_repository = $invoice_items_repository;
        $this->user_repository = $user_repository;
        $this->payment_methods_repository = $payment_methods_repository;
        $this->token_storage = $token_storage;
        $this->accounts_service = $accounts_service;
        $this->entity_manager = $entity_manager;
        $this->app_settings = $app_settings;
        $this->system_options_service = $system_options_service;
        $this->fee_scheduler_service = $fee_scheduler_service;
        $this->class_fee_schedule_repository = $class_fee_schedule_repository;
        $this->student_fee_service = $student_fee_service;
    }



    /**
     * Invoices
     * */

    ## validate before saving invoice.
    public function validate_invoice(ParameterBag $request)
    {
        $response = [];

        ## for which user invoice is created - invoice generated for.
        ## before that check, user is system user or walk in user.
        if (empty($request->get('u____s'))) {
            $response = $this->default_function->push_error(
                $response,
                'Please select user is system user or un-register user'
            );
        } else {
            if ($request->get('u____s') == 'w__') {
                ## if user is walk in then validate that, walk-in user name is given or not.
                if (empty($request->get('w__u_n'))) {
                    $response = $this->default_function->push_error($response, 'Please enter walk in user name.');
                }
            } else {
                ## user is system register user.
                if (empty($request->get('s__u_n'))) {
                    ## if system user is null or empty then return back to fill out the required fields.
                    $response = $this->default_function->push_error($response, 'Please enter system register username');
                }
            }
        }

        ## invoice generation time.
        if (empty($request->get('inv___g_t'))) {
            $response = $this->default_function->push_error(
                $response,
                'Invoice generation date & time should not be null or empty'
            );
        }

        ## invoice due date.
        if (!empty($request->get('d__dte')) && (!empty($request->get('inv___g_t')))) {
            ## validate enter date is past date or future date.
            if (strtotime($request->get('d__dte')) < strtotime(date('m-d-y'))) {
                $response = $this->default_function->push_error($response, 'Due date is not selected as past date.');
            }
        }

        ## if empty payment method.
        if (empty($request->get('p___mod'))) {
            $response = $this->default_function->push_error($response, 'Please select payment methods');
        }

        ## validate notes.
        if (empty($request->get('n___t'))) {
            $response = $this->default_function->push_error($response, 'Please enter notes types');
        } else {
            if ($request->get('n___t') == 's___t' && empty($request->get('s___t'))) {
                $response = $this->default_function->push_error($response, 'Please enter system notes.');
            } else {
                $response = $this->default_function->push_error($response, 'Please enter notes.');
            }
        }

        ## if user like to update the record.
        if (!empty($request->get('inv___ifID'))) {
            ## validate if invoice is already exits or not.
            if (!empty($request->get('inv__ref'))) {
                $inv_repo_response = $this->invoices_repository->findOneBy(
                    ['inv_reference' => $request->get('inv__ref')]
                );
                if ($inv_repo_response instanceof InvoicesRepository) {
                    $response = $this->default_function->push_error(
                        $response,
                        'Invoice with this '.$inv_repo_response->getInvReference().' reference is already exits.'
                    );
                }
            }
        }

        return $response;

    }

    ## add or update the invoices.
    public function add_edit_invoice(ParameterBag $request)
    {
        $response = [];

        ## validate invoice
        if (empty($invoice__validate__response = $this->validate_invoice($request))) {
            return $invoice__validate__response;
        }

        $Entity = null;
        if (empty($request->get('inv___ifID'))) {
            ## user is adding invoice
            $Entity = new Invoices();
        } else {
            ## user is updating invoice
            try {
                $Entity = $this->get__the__invoice($request->get('inv___ifID'));
            } catch (\Exception $exception) {
            }
        }

        ## if entity is not part of invoice then create an invoice instance.
        if (!$Entity instanceof Invoices) {
            $Entity = new Invoices();
        }

        if ($request->get('u____s') == 'w__' && !empty($request->get('w__u_n'))) {
            ## save invoice for the un-register user.
            $Entity->setInvoiceGeneratedForUnregisterUsername($request->get('w__u_n'));
        } else {
            ## get the user object.
            $user__object = $this->user_repository->find($request->get('s__u_n'));
            if ($user__object instanceof User) {
                ## user is valid user, save it - its mean this invoice is generated for this user, he will pay for this invoice.
                $Entity->setInvGeneratedFor($user__object);
            } else {
                $response = $this->default_function->push_error($response, 'Invalid user reference');
            }
        }

        ## if error then send back.
        if (!empty($response)) {
            return $response;
        }

        ## log - invoice is generated by which user, it may or may not be system.
        if ($request->get('in___gn__by') == 'r__u') {
            ## invoice is generated by register user
            ## this is current logged in user.
            $Entity->setInvGeneratedBy($this->token_storage->getToken()->getUser());
        }

        ## invoice is generated by non-register user.
        ## save the system notes - to know the purpose of generating this report.
        $Entity->setInvSystemNotes($request->get('s___t'));

        ## if error then send back.
        if (!empty($response)) {
            return $response;
        }

        ## save the payment method.
        ## before saving validate the payment method.


        if ($request->get('p___mod') == 'DPM') {
            ## set default payment method
            $payment__method = $this->accounts_service->get__the_payment_method(null, 'pm_is_default', true);
        } else {
            ## set payment method which user selects
            $payment__method = $this->accounts_service->get__the_payment_method($request->get('p___mod'));
        }

        if (!$payment__method instanceof PaymentMethods) {
            $response = $this->default_function->push_error($response, 'Payment method not found or no one Selected as Default Payment Method');
        } else {
            $Entity->setPaymentMethod($payment__method);
        }

        ## save the user notes
        $Entity->setInvUserNotes($request->get('u__n_t'));
        ## save generation date.
        $Entity->setInvGeneratedDatetime(
            $this->default_function->convert_datetimeStringIntoObject($request->get('inv___g_t'))
        );
        ## save due date.
        $Entity->setInvDueDate($this->default_function->convert_datetimeStringIntoObject($request->get('d__dte')));
        ## save invoice reference.
        $Entity->setInvReference($this->autoGeneratedTransaction($request->get('inv__ref'), $Entity->getId()));
        ## save class session enrolment reference
        $Entity->setClassSessionEnrolment($request->get('classSessionEnrolmentRf'));

        ## if error then send back.
        if (!empty($response)) {
            return $response;
        }

        ## save data in the database.
        try {
            $this->entity_manager->persist($Entity);
            $this->entity_manager->flush();

            ## save the invoice items.
            for ($x = 0; $x < count($request->get('inv__it__des')); $x++) {

                $bag = new ParameterBag();
                ## invoice id  - if user want to udpate the invoice item
                $bag->set(
                    'inv_it_ref',
                    !empty($request->get('inv_it_ref__')[$x]) ? $request->get('inv_it_ref__')[$x] : 0
                );
                ## set the invoice item description
                $bag->set(
                    'inv_it_des',
                    !empty($request->get('inv__it__des')[$x]) ? $request->get('inv__it__des')[$x] : ''
                );
                ## set the invoice item amount
                $bag->set(
                    'inv_it_am',
                    !empty($request->get('inv__it__am')[$x]) ? $request->get('inv__it__am')[$x] : ''
                );
                ## set the invoice item discount amount
                $bag->set(
                    'inv_it_d_am',
                    !empty($request->get('inv__it__dis__am')[$x]) ? $request->get('inv__it__dis__am')[$x] : ''
                );
                ## set the invoice item discount description
                $bag->set(
                    'inv_it_d_des',
                    !empty($request->get('inv__it__dis__des')[$x]) ? $request->get('inv__it__dis__des')[$x] : ''
                );
                ## set the invoice reference
                $bag->set('inv_refer', $Entity);

                $this->add_edit_invoice_items($bag);
            }
            if ($request->get('getLastIDWithOKMessage')) {
                $response = 'OK'.$Entity->getId();
            } else {
                if (empty($request->get('inv___ifID'))) {
                    ## when adding new record send okay, in this case we will reset the form
                    $response = 'OK';
                } else {
                    ## when updating existing record send this response, because i don't have to reset the form.
                    $response = 'dontResetForm_'.$Entity->getId();
                }
            }

        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }


        return $response;

    }

    ## validate the
    public function validateGenerateInvoiceWhileStudentEnrolmentInClass(ParameterBag $bag)
    {
        $response = [];
        ## Check if class id is missing.
        if (empty($bag->get('class'))) {
            $response = $this->default_function->push_error($response, 'Class Reference should not be null');
        }

        ## Check if user id is missing.
        if (empty($bag->get('studentRf'))) {
            $response = $this->default_function->push_error($response, 'User Reference should not be null');
        }

        return $response;
    }

    ## Generate invoice at time of student enrolment
    ## In paid to we will take user.
    public function generateInvoiceWhileStudentEnrolmentInClass(ParameterBag $bag)
    {
        $response = [];

        if (!empty($validationResponse = $this->validateGenerateInvoiceWhileStudentEnrolmentInClass($bag))) {
            return $validationResponse;
        }

        ## when student enroll in the class, he/she must have the class id, so first get the Class Fee from Class fee scheduler table, with the reference of class
        $ClassFeeScheduler = null;
        try {
            $ClassFeeScheduler = $this->class_fee_schedule_repository->getAdmissionClassFee((int)$bag->get('class'));
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        ## check the class scheduler instance
        if (empty($ClassFeeScheduler)) {
            $response = $this->default_function->push_error($response, 'Fee is not found against this class');
        }

        ## if error is not captured then throw an error.
        if (!empty($response)) {
            return $response;
        }

        ##
        $Bag_ = new ParameterBag();
        $Bag_->set('u____s', 'r__u');
        $Bag_->set('s__u_n', $bag->get('studentRf'));
        $Bag_->set('inv___g_t', '');
        $Bag_->set('d__dte', '');
        $Bag_->set('in___gn__by', 'r__u');
        $Bag_->set('p___mod', 'DPM');
        $Bag_->set('n___t', 's___t');
        $Bag_->set('s___t', 'Invoice generated by system, when admin enroll the student');
        $Bag_->set('classSessionEnrolmentRf', $bag->get('classSessionEnrolmentRf'));

        ## adding invoice items and description as the name of fee.
        $inv_it_des = [];
        $inv_it_am = [];

        /** @var ClassFeeSchedule $CFS */
        foreach ($ClassFeeScheduler as $CFS) {
            array_push($inv_it_am, $CFS->getAmount());
            array_push($inv_it_des, $CFS->getTitle());
        }


        $Bag_->set('inv__it__des', $inv_it_des);
        $Bag_->set('inv__it__am', $inv_it_am);

        return $this->add_edit_invoice($Bag_);

    }



    ## auto generate transaction.
    ## if user enter some invoice  number, system will keep it and check is this in database.
    ## if is in database, then we will keep of autoGenerated number otherwise we will keep user generated number.
    public function autoGeneratedTransaction($InvRef, $whereNoINID = null)
    {
        ## invoice prefix from app settings.
        $invoiceNoPreFix = (string)$this->app_settings->getDataFromAppSettings('invoicePreFix', true, 'invoice__re');
        $InvNo = 0;

        if (!empty($InvRef)) {
            try {


                ## findNotOneBy
                if (empty($whereNoINID)) {
                    ##
                    $InvRef = $invoiceNoPreFix.$InvRef;
                    $Invoice = $this->get__the__invoice(null, 'inv_reference', $InvRef);
                } else {
                    $Invoice = $this->invoices_repository->findNotOneBy('inv_reference', $InvRef, $whereNoINID);
                }
                ## if we found the invoice with the same reference.
                if (!$Invoice instanceof Invoices) {
                    ## user given invoice if user entered invoice is not found in the database.
                    $InvNo = $InvRef;
                }

            } catch (\Exception $exception) {
            }

        }
        ## if empty invoice no then suggested by us in the sequence using by option table.
        if (empty($InvNo)) {
            ## this will find in the database if, we found the record it will increase record by one.
            $InvNo = $this->system_options_service->getOption('lastInvoiceNo', 0);

            if ($InvNo instanceof Options) {
                $InvNo = (int)$InvNo->getOptionValue() + 1;
            } else {
                $InvNo = 1;
            }

            ## save updated last invoice no in the database.
            $bag = new ParameterBag();
            $bag->set('optNm', 'lastInvoiceNo');
            $bag->set('optVl', $InvNo);
            $this->system_options_service->saveOption($bag, true);


            ## creating invoice no.
            $InvNo = $invoiceNoPreFix.$InvNo;

        }

        return $InvNo;
    }

    ## get the  invoice
    public function get__the__invoice($reference = null, $key = null, $value = null)
    {
        $response = null;
        try {
            if (!empty($key) && !empty($value)) {
                $response = $this->invoices_repository->findOneBy([$key => $value]);
            } else {
                $response = $this->invoices_repository->find($reference);
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    ## validate invoice is paid or not.
    public function is__invoice_is_paid(int $InvID)
    {
        $invoice = $this->get__the__invoice($InvID);
        if (!empty($invoice)) {
            $invoice_items = $invoice->getTransactions()->toArray();
        }

        return $invoice;
    }

    ## get the defaulter report.
    ## the report of those men, who didn't pay before the due date.
    public function get__the__defaulterReport()
    {
        $bag = new ParameterBag();
        $bag->set('invStatus', 'defaulter');

        return $this->invoices_repository->search__inv($bag);
    }


    /**
     * Invoice items.
     * */

    ## get the  invoice
    public function get__the__invoiceItem($reference)
    {
        $response = null;
        if (!empty($reference)) {
            try {
                $response = $this->invoice_items_repository->find($reference);
            } catch (\Exception $exception) {
                $response = $exception->getMessage();
            }
        }

        return $response;
    }

    ## validate invoice items.
    public function validate_invoice_items(ParameterBag $request)
    {
        $response = [];

        ## if empty invoice item amount
        if (empty($request->get('inv_it_am'))) {
            $response = $this->default_function->push_error($response, 'Please enter invoice amount');
        }

        if (empty($request->get('inv_refer'))) {
            $response = $this->default_function->push_error($response, 'Please enter the invoice reference');
        }

        return $response;
    }

    ## add or edit invoice items.

    /**
     * @param ParameterBag $request
     *
     * @return array|mixed|string
     */
    public function add_edit_invoice_items(ParameterBag $request)
    {
        $response = [];

        ## validate invoice items.
        if ($validation_response = $this->validate_invoice_items($request)) {
            return $validation_response;
        }


        $Entity = null;
        if (empty($request->get('inv_it_ref'))) {
            $Entity = new InvoiceItems();
        } else {
            try {
                $Entity = $this->get__the__invoiceItem($request->get('inv_it_ref'));
            } catch (\Exception $exception) {
            }
        }

        if (!$Entity instanceof InvoiceItems) {
            $Entity = new InvoiceItems();
        }

        ## set item desc
        $Entity->setInvItemDescription($request->get('inv_it_des'));
        ## set item amount
        $Entity->setInvItemAmount((int)$request->get('inv_it_am'));
        ## set item discount
        $Entity->setInvItemDiscount((int)$request->get('inv_it_d_am'));
        ## set item discount description
        $Entity->setInvItemDiscountDescription($request->get('inv_it_d_des'));

        ## save the invoice reference
        if ($request->get('inv_refer') instanceof Invoices) {
            $Entity->setInvoiceReference($request->get('inv_refer'));
        } else {
            $invoice_response = $this->get__the__invoice($request->get('inv_refer'));
            if (!$invoice_response instanceof Invoices) {
                $response = $this->default_function->push_error($response, $invoice_response);
            }
        }

        ## if error then send back.
        if (!empty($response)) {
            return $response;
        }


        try {
            $this->entity_manager->persist($Entity);
            $this->entity_manager->flush();
            $response = 'OK';
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }


    ## Delete invoice items
    public function deleteInvoiceItem(int $invoiceRef)
    {
        ## get the invoice
        $invoiceItem = $this->get__the__invoiceItem($invoiceRef);
        if ($invoiceItem instanceof InvoiceItems) {
            $this->entity_manager->remove($invoiceItem);
            $this->entity_manager->flush();
            $invoiceItem = 'OK';
        }

        return $invoiceItem;
    }
}
