<?php


namespace App\Service\Students\Grade;


use App\Entity\Classes;
use App\Entity\ClassGrade;
use App\Entity\ClassGradeTemplate;
use App\Entity\ClassTemplate;
use App\Entity\GradeCategories;
use App\Entity\StudentGrades;
use App\Repository\ClassGradeRepository;
use App\Repository\ClassGradeTemplateRepository;
use App\Repository\GradeCategoriesRepository;
use App\Repository\StudentGradesRepository;
use App\Service\DefaultFunction;
use Doctrine\ORM\EntityManagerInterface;
use PhpParser\Node\Param;
use Proxies\__CG__\App\Entity\ClassSubjectTemplate;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\Validator\ConstraintViolation;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class GradeBase extends GradeCategoriesService
{


    /**
     * @var GradeCategoriesRepository
     */
    private $gradeCategoriesRepository;
    /**
     * @var DefaultFunction
     */
    private $defaultFunction;
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;
    /**
     * @var StudentGradesRepository
     */
    private $studentGradesRepository;
    /**
     * @var ValidatorInterface
     */
    private $validator;
    /**
     * @var ClassGradeRepository
     */
    private $classGradeRepository;
    /**
     * @var ClassGradeTemplateRepository
     */
    private $classGradeTemplateRepository;


    public function __construct(
        GradeCategoriesRepository $gradeCategoriesRepository,
        EntityManagerInterface $entityManager,
        DefaultFunction $defaultFunction,
        StudentGradesRepository $studentGradesRepository,
        ValidatorInterface $validator,
        ClassGradeRepository $classGradeRepository,
        ClassGradeTemplateRepository $classGradeTemplateRepository
    ) {
        parent::__construct($gradeCategoriesRepository, $entityManager);
        $this->gradeCategoriesRepository = $gradeCategoriesRepository;
        $this->defaultFunction = $defaultFunction;
        $this->entityManager = $entityManager;
        $this->studentGradesRepository = $studentGradesRepository;
        $this->validator = $validator;
        $this->classGradeRepository = $classGradeRepository;
        $this->classGradeTemplateRepository = $classGradeTemplateRepository;
    }

    /**
     * @param ParameterBag $bag
     * @return array|mixed|string|ConstraintViolationListInterface
     * Purpose: Save Grades to StudentGrade(Grade Base) table
     */
    public function saveGrades(ParameterBag $bag)
    {
        $studentGrade = $this->getGrade($bag->get('gradeItemRef'));
        if (!$studentGrade instanceof StudentGrades) {
            $studentGrade = new StudentGrades();
        }

        ## get grade category reference
        $gradeCategory = $this->getGradeCategory($bag->get('gradeCat'));
        if ($gradeCategory instanceof GradeCategories) {
            ## set grade category reference
            $studentGrade->setStudentGradeCategory($gradeCategory);
        }

        ## set grade description
        $studentGrade->setStudentGradeDesc($bag->get('gradeDesc', ''));
        ## grade applicable when marks are equal or greater than
        $studentGrade->setStudentGradeEqualOrGreaterThanMarks($bag->get('gradeNumberEqualOrGreater', 0));
        ## set grade name
        $studentGrade->setStudentGradeName($bag->get('gradeName', ''));

        $errors = $this->validator->validate($studentGrade);
        $errors = $this->defaultFunction->parseConstraintViolationInString($errors);
        if (empty($errors)) {

            ## Duplication Check
            $duplicationExists = $this->studentGradesRepository->gradeCategoryGradeNameGradeMarksDuplicationCheck(
                $gradeCategory->getId(),
                $bag->get('gradeName'),
                $bag->get('gradeNumberEqualOrGreater')
            );

            if ($duplicationExists) {
                ## if records are duplication
//                $errors = 'Trying to save duplicate record';
                $errors = 'OK';
            } else {
                ## if records are not duplication then save
                try {
                    $this->entityManager->persist($studentGrade);
                    $this->entityManager->flush();
                    $errors = 'OK';
                } catch (\Exception $exception) {
                    $errors = $exception->getMessage();
                }
            }
        }

        return $errors;
    }

    /**
     * @param ParameterBag $bag
     * @return array|mixed
     * Purpose: Save Bulk Grades - this method filter data and provide to require method.
     */
    public function saveBulkGrades(ParameterBag $bag)
    {
        $response = [];
        if (!empty($bag->get('gradeName')) && array($bag->get('gradeName'))) {
            for ($x = 0; $x < count($bag->get('gradeName')); $x++) {
                $bag2 = new ParameterBag();

                ## grade reference if want to edit
                $bag2->set(
                    'gradeItemRef',
                    !empty($bag->get('gradeItemRef')[$x]) ? $bag->get('gradeItemRef')[$x] : null
                );
                ## grade name
                $bag2->set('gradeName', !empty($bag->get('gradeName')[$x]) ? $bag->get('gradeName')[$x] : '');
                ## grade number if equal or greater than
                $bag2->set(
                    'gradeNumberEqualOrGreater',
                    !empty($bag->get('gradeNumberEqualOrGreater')[$x]) ? $bag->get(
                        'gradeNumberEqualOrGreater'
                    )[$x] : 0
                );
                ## grade description
                $bag2->set('gradeDesc', !empty($bag->get('gradeDesc')[$x]) ? $bag->get('gradeDesc')[$x] : '');
                ## grade category
                $bag2->set('gradeCat', $bag->get('gradeCat'));
                ## save grades.
                $response = $this->defaultFunction->push_error($response, $this->saveGrades($bag2));
            }
        }

        ## get the error or success response from bulk action
        return $this->defaultFunction->purify_success_error_response_in_bulk_operation($response);

    }

    /**
     * @param null $id
     * Purpose: get grades Object
     *
     * @param null $key
     * @param null $value
     * @return StudentGrades|string|null
     */
    public function getGrade($id = null, $key = null, $value = null)
    {
        $response = null;
        try {
            if (!empty($id)) {
                $response = $this->studentGradesRepository->find($id);
            } else {
                $response = $this->studentGradesRepository->findBy([$key => $value]);
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    public function copyGradeToClassGradeTemplate(
        GradeCategories $GradeCategory,
        ClassTemplate $ClassTemplate,
        Classes $classes
    ) {
        $response = [];
        $studentGrades = $this->getGrade(null, 'student_grade_category', $GradeCategory);
//        dd($studentGrades);
        /** @var StudentGrades $grade */
        foreach ($studentGrades as $grade) {
            try {
                $ClassGradeTemplate = new ClassGradeTemplate();
                $ClassGradeTemplate->setStudentGradeCategory($GradeCategory);
                $ClassGradeTemplate->setClassTemplate($ClassTemplate);
                $ClassGradeTemplate->setGradeName($grade->getStudentGradeName());
                $ClassGradeTemplate->setGradeDesc($grade->getStudentGradeDesc());
                $ClassGradeTemplate->setMarksEqualOrGreaterThan($grade->getStudentGradeEqualOrGreaterThanMarks());
                $this->entityManager->persist($ClassGradeTemplate);
                $this->entityManager->flush();

                $this->copyGradeToClassGrade($GradeCategory, $classes, $ClassGradeTemplate, $grade);

            } catch (\Exception $exception) {
                $response = array_push($response, $exception->getMessage());
            }
        }

        return $response;
    }

    public function copyGradeToClassGrade(
        GradeCategories $GradeCategory,
        Classes $Class,
        ClassGradeTemplate $classGradeTemplate,
        StudentGrades $studentGrades
    ) {
        $response = [];
        try {
            $ClassGrade = new ClassGrade();
            $ClassGrade->setStudentGradeCategory($GradeCategory);
            $ClassGrade->setClassGradeTemplate($classGradeTemplate);
            $ClassGrade->setClass($Class);
            $ClassGrade->setClassGradeName($studentGrades->getStudentGradeName());
            $ClassGrade->setClassGradeDesc($studentGrades->getStudentGradeDesc());
            $ClassGrade->setClassMarksEqualOrGreaterThan($studentGrades->getStudentGradeEqualOrGreaterThanMarks());
            $this->entityManager->persist($ClassGrade);
            $this->entityManager->flush();
        } catch (\Exception $exception) {
            $response = array_push($response, $exception->getMessage());
        }

        return $response;


        $studentGrades = $this->getGrade(null, 'student_grade_category', $GradeCategory);
        /** @var StudentGrades $studentGrade */
        foreach ($studentGrades as $studentGrade) {
        }

        return $response;
    }

    ## assign Grades to Class
    ## Procedure:
    ## 1. Check If any grade assigned to Class? If grade assigned to Class, then Delete the Grades first.
    ## 2. If Grade is not assigned to Class, then First assign grade to ClassGradeTemplate & then ClassGrade
    public function assignGradesToClass($GradeCategory, Classes $classes)
    {
        ## get Grade Category
        $GradeCategory = $this->getGradeCategory($GradeCategory);
        if ($GradeCategory instanceof GradeCategories) {
            ## delete all grade which is attached to Class
            $this->classGradeRepository->deleteGradeOnBaseOfClass($classes->getId());
            ## delete all grade which is attached to ClassGradeTemplate
            $this->classGradeTemplateRepository->deleteGradeOnBaseOfClassTemplate(
                $classes->getClassTemplate()->getId()
            );
            $this->copyGradeToClassGradeTemplate($GradeCategory, $classes->getClassTemplate(), $classes);
        }
    }

}
