<?php


namespace App\Service\Students\Grade;


use App\Entity\ClassGradeTemplate;
use App\Entity\ClassTemplate;
use App\Entity\GradeCategories;
use App\Entity\StudentGrades;
use App\Repository\ClassGradeRepository;
use App\Repository\ClassGradeTemplateRepository;
use App\Repository\ClassTemplateRepository;
use App\Repository\GradeCategoriesRepository;
use App\Repository\StudentGradesRepository;
use App\Service\DefaultFunction;
use App\Service\StudentClasses;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;

class ClassGradeTemplateService extends GradeBase
{

    /**
     * @var GradeCategoriesRepository
     */
    private $gradeCategoriesRepository;
    /**
     * @var StudentClasses
     */
    private $studentClasses;
    /**
     * @var ClassTemplateRepository
     */
    private $classTemplateRepository;
    /**
     * @var ValidatorInterface
     */
    private $validator;
    /**
     * @var DefaultFunction
     */
    private $defaultFunction;
    /**
     * @var EntityManagerInterface
     */
    private $entityManager;
    /**
     * @var StudentGradesRepository
     */
    private $studentGradesRepository;
    /**
     * @var ClassGradeTemplateRepository
     */
    private $classGradeTemplateRepository;


    public function __construct(
        GradeCategoriesRepository $gradeCategoriesRepository,
        EntityManagerInterface $entityManager,
        DefaultFunction $defaultFunction,
        StudentGradesRepository $studentGradesRepository,
        ValidatorInterface $validator,
        StudentClasses $studentClasses,
        ClassGradeTemplateRepository $classGradeTemplateRepository,
        ClassGradeRepository $classGradeRepository
    ) {
        parent::__construct(
            $gradeCategoriesRepository,
            $entityManager,
            $defaultFunction,
            $studentGradesRepository,
            $validator,
            $classGradeRepository,
            $classGradeTemplateRepository
        );
        $this->gradeCategoriesRepository = $gradeCategoriesRepository;
        $this->studentClasses = $studentClasses;
        $this->validator = $validator;
        $this->defaultFunction = $defaultFunction;
        $this->entityManager = $entityManager;
        $this->studentGradesRepository = $studentGradesRepository;
        $this->classGradeTemplateRepository = $classGradeTemplateRepository;
    }

    /**
     * Purpose: Copy student grades from grade base table to class template
     * @param int $studentGradeCategoryRef
     * @param ClassTemplate $classTemplate
     * @return string
     */
    public
    function copyStudentDefinedGradeToClassTemplateFromStudentGrade(
        int $studentGradeCategoryRef,
        ClassTemplate $classTemplate
    ) {
        $response = [];
        $gradeCategory = $this->getGradeCategory($studentGradeCategoryRef);
        if ($gradeCategory instanceof GradeCategories) {
            ## if grade category found.
            $listOfDefinedGrades = $gradeCategory->getStudentGrades()->toArray();
            /** @var StudentGrades $value */
            foreach ($listOfDefinedGrades as $key => $value) {
                if ($value instanceof StudentGrades) {
                    $bag = new ParameterBag();
//                    $bag->set('gradeItemRef', $value->getId());
                    $bag->set('gradeCategoryInstance', $gradeCategory);
                    $bag->set('gradeDesc', $value->getStudentGradeDesc());
                    $bag->set('gradeNumberEqualOrGreater', $value->getStudentGradeEqualOrGreaterThanMarks());
                    $bag->set('gradeName', $value->getStudentGradeName());
                    $bag->set('classTemplateInstance', $classTemplate);
                    $response = $this->defaultFunction->push_error($response, $this->saveGradeIntoClassTemplate($bag));
                }
            }
        }

        return $this->defaultFunction->purify_success_error_response_in_bulk_operation($response);
    }

    /**
     * Purpose: Save Grades into Class Grade Template
     * @param ParameterBag $bag
     * @return array|mixed|string|ConstraintViolationListInterface
     */
    public
    function saveGradeIntoClassTemplate(
        ParameterBag $bag
    ) {

        $classGradeTemplate = $this->getClassGradeTemplate((int)$bag->get('gradeItemRef'));
        if (!$classGradeTemplate instanceof ClassGradeTemplate) {
            $classGradeTemplate = new ClassGradeTemplate();
        }

        ## get grade category
        ## 1. check is grade instance is given.
        if ($bag->get('gradeCategoryInstance') instanceof GradeCategories) {
            $gradeCategory = $bag->get('gradeCategoryInstance');
        } else {
            $gradeCategory = $this->getGradeCategory($bag->get('gradeCat'));
        }

        ## If grade Category has found then save.
        if ($gradeCategory instanceof GradeCategories) {
            ## set grade category reference
            $classGradeTemplate->setStudentGradeCategory($gradeCategory);
        }

        ## set the Class Ref
        if ($bag->get('classTemplateInstance') instanceof ClassTemplate) {
            $classTemplate = $bag->get('classTemplateInstance');
        } else {
            $classTemplate = $this->studentClasses->getClassTemplate($bag->get('classTemplate'));
        }

        ## If class template has found then save.
        if ($classTemplate instanceof ClassTemplate) {
            ## set class template reference
            $classGradeTemplate->setClassTemplate($classTemplate);
        }

        ## set grade description
        $classGradeTemplate->setGradeDesc($bag->get('gradeDesc', ''));
        ## grade applicable when marks are equal or greater than
        $classGradeTemplate->setMarksEqualOrGreaterThan($bag->get('gradeNumberEqualOrGreater', 0));
        ## set grade name
        $classGradeTemplate->setGradeName($bag->get('gradeName', ''));

        $errors = $this->validator->validate($classGradeTemplate);
        $errors = $this->defaultFunction->parseConstraintViolationInString($errors);

        if (empty($errors)) {

            ## Duplication Check
            $duplicationExists = $this->classGradeTemplateRepository->gradeCategoryGradeNameGradeMarksDuplicationCheck(
                $gradeCategory->getId(),
                $bag->get('gradeName'),
                $bag->get('gradeNumberEqualOrGreater'),
                $classTemplate->getId()
            );

            if ($duplicationExists) {
                ## if records are duplication
//                $errors = 'Trying to save duplicate record';
                $errors = 'OK';
            } else {
                ## if records are not duplication then save
                try {
                    $this->entityManager->persist($classGradeTemplate);
                    $this->entityManager->flush();
                    $errors = 'OK';
                } catch (\Exception $exception) {
                    $errors = $exception->getMessage();
                }
            }
        }

        return $errors;
    }


    /**
     * @param ParameterBag $bag
     * @return array|mixed
     * Purpose: Save Bulk ClassTemplateGrades - this method filter data and provide to require method.
     */
    public function saveBulkGradeIntoClassTemplate(ParameterBag $bag)
    {
        $response = [];
        if (!empty($bag->get('gradeName')) && array($bag->get('gradeName'))) {
            for ($x = 0; $x < count($bag->get('gradeName')); $x++) {
                $bag2 = new ParameterBag();

                ## grade reference if want to edit
                $bag2->set(
                    'gradeItemRef',
                    !empty($bag->get('gradeItemRef')[$x]) ? $bag->get('gradeItemRef')[$x] : null
                );
                ## grade name
                $bag2->set('gradeName', !empty($bag->get('gradeName')[$x]) ? $bag->get('gradeName')[$x] : '');
                ## grade number if equal or greater than
                $bag2->set(
                    'gradeNumberEqualOrGreater',
                    !empty($bag->get('gradeNumberEqualOrGreater')[$x]) ? $bag->get(
                        'gradeNumberEqualOrGreater'
                    )[$x] : 0
                );
                ## grade description
                $bag2->set('gradeDesc', !empty($bag->get('gradeDesc')[$x]) ? $bag->get('gradeDesc')[$x] : '');
                ## grade category
                $bag2->set('gradeCat', $bag->get('gradeCat'));
                ## class template id
                $bag2->set('classTemplate', $bag->get('classTemplate'));
                ## save grades.
                $response = $this->defaultFunction->push_error($response, $this->saveGradeIntoClassTemplate($bag2));
            }
        }

        ## get the error or success response from bulk action
        return $this->defaultFunction->purify_success_error_response_in_bulk_operation($response);
    }


    /**
     * @param null $gradeTemplateId
     * @param null $key
     * @param null $value
     * @return ClassTemplate[]|mixed|object|string|null
     * Purpose Get the Class Grade Template
     */
    public
    function getClassGradeTemplate(
        $gradeTemplateId = null,
        $key = null,
        $value = null
    ) {
        $response = null;
        try {
            if (!empty($gradeTemplateId)) {
                $response = $this->classGradeTemplateRepository->find($gradeTemplateId);
            } else {
                if (!empty($key) && !empty($value)) {
                    $response = $this->classGradeTemplateRepository->findBy([$key => $value]);
                }
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }
}
