<?php

namespace App\Repository;

use App\Entity\Invoices;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\NoResultException;
use Symfony\Bridge\Doctrine\RegistryInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\Validator\Constraints\Date;

/**
 * @method Invoices|null find($id, $lockMode = null, $lockVersion = null)
 * @method Invoices|null findOneBy(array $criteria, array $orderBy = null)
 * @method Invoices[]    findAll()
 * @method Invoices[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class InvoicesRepository extends ServiceEntityRepository
{

    public function __construct(RegistryInterface $registry)
    {
        parent::__construct($registry, Invoices::class);
    }

    ## Search invoice in the database.
    public function search__inv(\Symfony\Component\HttpFoundation\ParameterBag $request)
    {
        $builder = $this->createQueryBuilder('invoices');
        $param_pusher = [];

        ## left join with invoice items
        $builder->leftJoin('invoices.invoiceItems', 'invoiceItems')->addSelect('invoiceItems');
        ## left join with transactions.
        $builder->leftJoin('invoices.transactions', 'transactions')->addSelect('transactions');
        ## left join with the user
        $builder->leftJoin('invoices.inv_generated_by', 'userInvoiceGeneratedFor')
            ->addSelect('userInvoiceGeneratedFor');

        ## if logged in user is  Guardian then show only attached students record
        if ($request->get('isGuardian')) {
            if (empty($request->get('guardianStudents'))) {
                return [];
            } else {
                $builder->add(
                    'where',
                    $builder->expr()->in('invoices.inv_generated_for', $request->get('guardianStudents'))
                );
            }
        }

        ## search by id.
        if (!empty($request->get('id'))) {
            $builder->andwhere('invoices.id = :inv_id')->setParameter('inv_id', $request->get('id'));
        }
        ## search by user - generated for user.
        if (!empty($request->get('user'))) {
            $builder->andwhere('invoices.inv_generated_for = :user')->setParameter('user', $request->get('user'));
        }
        ## search by invoice due date
        if (!empty($request->get('DueDateRange'))) {
            $invoiceDueDate = explode(' / ', $request->get('DueDateRange'));
            if (is_array($invoiceDueDate) && !empty($invoiceDueDate[0] && !empty($invoiceDueDate[1]))) {
                $builder->andWhere('invoices.inv_due_date BETWEEN :From and :To');
                $builder->setParameter('From', date('Y-m-d', strtotime(trim($invoiceDueDate[0]))).'%');
                $builder->setParameter('To', date('Y-m-d', strtotime(trim($invoiceDueDate[1]))).'%');
            }
        }
        ## search by invoice reference
        if (!empty($request->get('refer'))) {
            $builder->andwhere('invoices.inv_reference= :inv__ref')
                ->setParameter('inv__ref', $request->get('refer'));
        }
        ## Search by the user notes.
        if (!empty($request->get('userNotes'))) {
            $builder->andwhere('invoices.inv_user_notes LIKE :invNotes')
                ->setParameter('invNotes', '%'.$request->get('userNotes').'%');
        }
        ## search by class
        if (!empty($request->get('inv__class'))) {
            $builder->leftJoin('invoices.class_session_enrolment', 'class_session_enrolment')
                ->andWhere('class_session_enrolment.class_session = :classSessionEnrolmentRef')
                //$builder->andwhere('invoices.class_session_enrolment = :classSessionEnrolmentRef')
                ->setParameter('classSessionEnrolmentRef', $request->get('inv__class'));
        }
        ## search invoice by status.
        if (!empty($request->get('invStatus'))) {
            if ($request->get('invStatus') == 'p') {
                ## search by paid
                $builder->andHaving('SUM(invoiceItems.inv_item_amount) = SUM(transactions.amount)');
            } else {
                if ($request->get('invStatus') == 'unp') {
                    ## search by un-paid
                    $builder->andWhere('transactions.amount IS NULL')->andWhere('invoiceItems.id IS NOT NULL');
                } else {
                    if ($request->get('invStatus') == 'pp') {
                        ## search by partially paid
                        $builder->andHaving('SUM(invoiceItems.inv_item_amount) > transactions.amount')
                            ->andWhere('transactions.amount is not NULL');
                    } else {
                        if ($request->get('invStatus') == 'defaulter') {
                            ## search by defaulter
                            return $this->defaulterInvoice($request);
                        }
                    }
                }
            }
        }

        return $builder->getQuery()->getResult();
    }

    ## find value in database except this one.
    public function findNotOneBy($key, $value, $whereNotIn)
    {
        try {
            return $this->createQueryBuilder('invoices')
                ->where(sprintf('invoices.%s= :value', $key))
                ->andWhere('invoices.id != :whereNotIn')
                ->setParameter('value', $value)
                ->setParameter('whereNotIn', $whereNotIn)
                ->getQuery()
                ->getSingleResult();
        } catch (NoResultException $e) {
        } catch (NonUniqueResultException $e) {
            return $e->getMessage();
        }
    }

    ## defaulter invoice.
    public function defaulterInvoice(ParameterBag $parameterBag)
    {

        $connection = $this->getEntityManager()->getConnection();

        ## main query
        $queryBuilder = $connection->createQueryBuilder()->select('*')->from('invoices');

        ## sub query for invoice items
        $invoiceItemSubQuery = $connection->createQueryBuilder()
            ->select('*,SUM(invoice_items.inv_item_amount) as InvoiceSum')
            ->from('invoice_items')
            ->groupBy('invoice_items.invoice_reference_id');


        ## sub query for the transactions.
        $transactionSubQuery = $connection->createQueryBuilder()
            ->select('*,SUM(transactions.amount) as TransactionSum')
            ->from('transactions')
            ->groupBy('transactions.invoice_reference_id');


        ## join with the invoice items
        $queryBuilder->leftJoin(
            'invoices',
            sprintf('(%s)', $invoiceItemSubQuery),
            'invoice_items',
            'invoices.id = invoice_items.invoice_reference_id'
        );
        ## join with the transactions
        $queryBuilder->leftJoin(
            'invoices',
            sprintf('(%s)', $transactionSubQuery),
            'transactions',
            'invoices.id = transactions.invoice_reference_id'
        );
        // join with user
        $queryBuilder->leftJoin('invoices', 'user', 'user', 'invoices.inv_generated_for_id = user.id');

        if ($parameterBag->get('overDue') == '1week') {
            $invoiceDueDate = '7';
        } elseif ($parameterBag->get('overDue') == '1month') {
            $invoiceDueDate = '30';
        } else {
            $invoiceDueDate = '1';
        }

        $queryBuilder->andWhere(
            "DATE(DATE_ADD(invoices.inv_due_date,  INTERVAL {$invoiceDueDate} DAY)) = CURRENT_DATE()"
        )
            ->andWhere('TransactionSum is null OR InvoiceSum > TransactionSum');

        $query = $queryBuilder->execute();

        return $query->fetchAll();

    }
}
