<?php


namespace App\Service\Fees;


use App\Entity\AppSettings;
use App\Entity\Classes;
use App\Entity\ClassFee;
use App\Entity\ClassFeeSchedule;
use App\Entity\ClassSessionEnrolment;
use App\Entity\ClassSessionEnrolmentFeeSchedule;
use App\Entity\ClassSubject;
use App\Entity\StudentFee;
use App\Repository\ClassesRepository;
use App\Repository\ClassFeeRepository;
use App\Repository\ClassFeeScheduleRepository;
use App\Repository\ClassSessionEnrolmentFeeScheduleRepository;
use App\Repository\StudentFeeRepository;
use App\Service\DefaultFunction;
use App\Service\StudentClasses;
use App\Service\StudentsEnrollment\Enrollments;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Twig\Environment;

class FeeSchedulerService
{

    /**
     * @var DefaultFunction
     */
    private $default_function;
    /**
     * @var EntityManagerInterface
     */
    private $entity_manager;
    /**
     * @var ClassFeeScheduleRepository
     */
    private $class_fee_schedule_repository;
    /**
     * @var ClassesRepository
     */
    private $classes_repository;
    /**
     * @var StudentFeeRepository
     */
    private $student_fee_repository;
    /**
     * @var StudentClasses
     */
    private $student_classes;
    /**
     * @var StudentFeeService
     */
    private $student_fee_service;
    /**
     * @var Enrollments
     */
    private $enrollments;
    /**
     * @var ClassSessionEnrolmentFeeScheduleRepository
     */
    private $class_session_enrolment_fee_schedule_repository;
    /**
     * @var ClassFeeRepository
     */
    private $class_fee_repository;
    /**
     * @var Environment
     */
    private $environment;
    /**
     * @var AppSettings
     */
    private $appSettings;

    public function __construct(
        Environment $environment,
        ClassFeeRepository $class_fee_repository,
        DefaultFunction $default_function,
        Enrollments $enrollments,
        ClassSessionEnrolmentFeeScheduleRepository $class_session_enrolment_fee_schedule_repository,
        StudentClasses $student_classes,
        StudentFeeService $student_fee_service,
        EntityManagerInterface $entity_manager,
        ClassFeeScheduleRepository $class_fee_schedule_repository,
        ClassesRepository $classes_repository,
        StudentFeeRepository $student_fee_repository,
        \App\Service\AppSettings $appSettings
    ) {
        $this->default_function = $default_function;
        $this->entity_manager = $entity_manager;
        $this->class_fee_schedule_repository = $class_fee_schedule_repository;
        $this->classes_repository = $classes_repository;
        $this->student_fee_repository = $student_fee_repository;
        $this->student_classes = $student_classes;
        $this->student_fee_service = $student_fee_service;
        $this->enrollments = $enrollments;
        $this->class_session_enrolment_fee_schedule_repository = $class_session_enrolment_fee_schedule_repository;
        $this->class_fee_repository = $class_fee_repository;
        $this->environment = $environment;
        $this->appSettings = $appSettings;
    }

    /**
     * Table:Class Fee Schedule
     *  * */

    ## validate class fee scheduler..
    public function validate__class_fee_scheduler(ParameterBag $request)
    {
        $response = [];

        ## name of generation fee.
        if (empty($request->get('tie'))) {
            $response = $this->default_function->push_error($response, 'Please type fee name');
        }
        ## generation date
        if (empty($request->get('gd')) && $request->get('aDTFee') <> '2') {
            $response = $this->default_function->push_error($response, 'Please type generation fee');
        }
        ## due date
        if (empty($request->get('dd')) && $request->get('aDTFee') <> '2') {
            $response = $this->default_function->push_error($response, 'Please type due date');
        }
        ## last due date
        if (empty($request->get('ldd')) && $request->get('aDTFee') <> '2') {
            $response = $this->default_function->push_error($response, 'Please type last due date');
        }
        ## get the class reference
        if (empty($request->get('c_r'))) {
            $response = $this->default_function->push_error($response, 'Please type Class reference');
        }

        return $response;
    }

    ## save fee scheduler in bulk
    public function save_class_fee_schedule_bulk(ParameterBag $request)
    {
        if (empty($request->get('tie')) || !is_array($request->get('tie'))) {
            return 'Fee name is missing.';
        }

        $response = [];

        ## save fee schedule one by one.
        for ($x = 0; $x < count($request->get('tie')); $x++) {
            $bag = new ParameterBag();

            ## get the name of the fee schedule
            $bag->set('tie', $request->get('tie')[$x]);
            ## fee scheduler reference
            $bag->set('f_s_r', $request->get('f_s_r')[$x]);
            ## amount of the fee
            $bag->set('am', $request->get('am')[$x]);
            ## generation time
            $bag->set('gd', $request->get('gd')[$x]);
            ## due date
            $bag->set('dd', $request->get('dd')[$x]);
            ## last  due date
            $bag->set('ldd', $request->get('ldd')[$x]);
            ## class reference
            $bag->set('c_r', $request->get('c_r'));
            ## admission time fee reference
            $bag->set('aDTFee', $request->get('admissionTime')[$x]);
            ## subject reference
            $bag->set('s_r', $request->get('subject__'));

            ## fee reference
            isset($request->get('f_r')[$x]) ? $fee__ = $request->get('f_r')[$x] : $fee__ = 0;
            $bag->set('f_r', $fee__);
            ## student fee reference
            isset($request->get('Sf_r')[$x]) ? $StudentFee__Ref = $request->get('Sf_r')[$x] : $StudentFee__Ref = 0;
            $bag->set('Sf_r', $StudentFee__Ref);
            ## class fee reference
            isset($request->get('Cf_r')[$x]) ? $ClassFee__Ref = $request->get('Cf_r')[$x] : $ClassFee__Ref = 0;
            $bag->set('Cf_r', $ClassFee__Ref);

            ## add class fee scheduler reference if we try to edit the class fee scheduler record.
            isset($request->get('CFS_r')[$x]) ? $ClassFeeSchedulerRef = $request->get(
                'CFS_r'
            )[$x] : $ClassFeeSchedulerRef = 0;
            $bag->set('CFS_r', $ClassFeeSchedulerRef);


            ## save the user reference
            if (empty($request->get('u_r'))) {
                ## save class fee schedule
                if (!empty($bag->get('s_r')) && !empty($bag->get('aDTFee'))) {
                    $re = 'OK';
                } else {
                    $re = $this->save_class_fee_schedule($bag);
                }
            } else {
                ## save special fee scheduler for special student
                $bag->set('u_r', $request->get('u_r'));
                $re = $this->_save_class_fee_schedule_for_special_student($bag);
            }

            $response = $this->default_function->push_error($response, $re);
        }

        return $response;
    }

    ## save fee scheduler
    public function save_class_fee_schedule(ParameterBag $request)
    {
        $response = [];
        ## check validate response.
        if (!empty($validation_response = $this->validate__class_fee_scheduler($request))) {
            return $validation_response;
        }

        ## add new Class Fee Scheduler
        $Entity = null;
        if (!empty($request->get('CFS_r'))) {
            ## Search in database
            $Entity = $this->getFeeFromClassFeeScheduler($request->get('CFS_r'));
        }

        ## if we not found data in database, then Create new entity
        if (!$Entity instanceof ClassFeeSchedule) {
            $Entity = new ClassFeeSchedule();
        }


        ## get the class.
        $Class = $this->student_classes->get_the_class($request->get('c_r'));
        if (!$Class instanceof Classes) {
            $response = $this->default_function->push_error($response, $Class);
        }

        ## get the class_subject object.
        $Class_subject = null;
        if ($request->get('s_r')) {
            $Class_subject = $this->entity_manager->getRepository(ClassSubject::class)->find($request->get('s_r'));
            if ($Class_subject instanceof ClassSubject) {
                $Entity->setClassSubject($Class_subject);
            }
        }

        ## set amount
        $Entity->setAmount((float)$request->get('am'));

        ## get the Class fee Ref reference if not empty
        if (!empty($request->get('Cf_r'))) {
            $Fee = $this->student_fee_service->getFeeFromClassFeeTable($request->get('Cf_r'));
            if (!$Fee instanceof ClassFee) {
                $response = $this->default_function->push_error($response, 'This kind of fee is not exits in database');
                $response = $this->default_function->push_error($response, $Fee);
            } else {
                ## save amount of fee
                $Entity->setAmount((float)$Fee->getAmount());
                ## save class fee
                $Entity->setClassFee($Fee);
            }
        }

        ## Get student fee
        if (!empty($request->get('Sf_r'))) {
            $Fee = $this->student_fee_service->getFeeFromStudentFeeTable($request->get('Sf_r'));
            if (!$Fee instanceof StudentFee) {
                $response = $this->default_function->push_error($response, 'This kind of fee is not exits in database');
                $response = $this->default_function->push_error($response, $Fee);
            } else {
                ## save amount of fee
                $Entity->setAmount((float)$Fee->getAmount());
                ## save class fee
                $Entity->setStudentFee($Fee);
            }
        }

        ## if error then send it back
        if (!empty($response)) {
            return $response;
        }

        $Entity->setTitle($request->get('tie'));
        $Entity->setClass($Class);
        $Entity->setGenerationDate($this->default_function->convert_datetimeStringIntoObject($request->get('gd')));
        $Entity->setDueDate($this->default_function->convert_datetimeStringIntoObject($request->get('dd')));
        $Entity->setLastDueDate($this->default_function->convert_datetimeStringIntoObject($request->get('ldd')));
        $Entity->setProcess(1);

        if (!$Class_subject instanceof ClassSubject) {
            ## set fee is admission time fee if it is. through this flag we will generate invoice at time of student admission.
            $Entity->setIsAdmissionFee($this->default_function->parse__boolean($request->get('aDTFee')));
        }

        try {
            $this->entity_manager->persist($Entity);
            $this->entity_manager->flush();
            $response = 'OK';
        } catch (\Exception $exception) {
            $response = $this->default_function->push_error($response, $exception->getMessage());
        }

        return $response;
    }

    ## get classFeeScheduler
    public function getFeeFromClassFeeScheduler(
        int $id = null,
        string $key = null,
        string $value = null,
        $findBy = 'findOneBy'
    ) {
        $response = null;
        try {
            if (!empty($id)) {
                $response = $this->class_fee_schedule_repository->find($id);
            } else {
                if (!empty($key) && !empty($value)) {
                    $response = $this->class_fee_schedule_repository->$findBy([$key => $value]);
                }
            }
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        return $response;
    }

    ## Load class fee on base of class
    ## this method will search fee against the class in the ClassFee table first, if found data in the ClassFee table it will server data from their, if we not found the result then system will get the result from ClassFee table and show Class Fee in the tution dropdown only.
    public function loadClassFeeOnBaseOfClass(int $classRef)
    {
        ## if class is empty
        if (empty($classRef)) {
            return 'Class not found';
        }

        ## check class existence in the Class_fee_schedule table.
        $ClassFeeScheduler = $this->getFeeFromClassFeeScheduler(null, 'class', $classRef, 'findBy');
        if (!empty($ClassFeeScheduler) && $ClassFeeScheduler[0] instanceof ClassFeeSchedule) {
            ## yeah, found in database & return back to the client.
            $response = 'escape__fromErro_replaceAllContent'.$this->environment->render(
                    'fee/ClassFeeScheduler/classFeeSchedulerEditInterface.html.twig',
                    [
                        'FeesList' => $ClassFeeScheduler,
                        'from_subject' => false,
                    ]
                );
        } else {
            ## not found in the database
            $response = 'escape__fromErro_'.$this->environment->render(
                    'fee/load_fee_on_based_of_classes.html.twig',
                    [
                        'classFee' => $this->class_fee_repository->findBy(['class' => $classRef]),
                    ]
                );
        }

        return $response;
    }

    ## delete fee from class fee scheduler table.
    public function deleteFeeFromClassFeeSchedulerTable(int $classFeeSchedulerRef)
    {
        ##
        $response = null;
        $classFeeSchedule = $this->getFeeFromClassFeeScheduler($classFeeSchedulerRef);
        if ($classFeeSchedule instanceof ClassFeeSchedule) {
            $this->entity_manager->remove($classFeeSchedule);
            $this->entity_manager->flush();
            $response = 'OK';
        } else {
            $response = 'No such fee found in class fee scheduler';
        }

        return $response;
    }

    ## get the all classes fee.
    public function getAllClassesFeeFromClassFeeScheduler()
    {

        ## get details from settings
        $showDeleteRecords = $this->appSettings->getAppSettingsFromSessions('show_deleted_records');

        return $this->class_fee_schedule_repository->listOfFees(
            $this->default_function->parse__YesNoToBoolean($showDeleteRecords)
        );
    }

    /**
     *  === Handling SubjectFeeScheduler -- System is scheduling fee on base of subject
     */

    /**
     *
     * Load fees on base of subjects
     * @param int $classRef
     * @param int $subjectRef
     * @return null
     * @throws \Twig\Error\LoaderError
     * @throws \Twig\Error\RuntimeError
     * @throws \Twig\Error\SyntaxError
     */
    public function loadSubjectFeeOnBaseOfClass(int $classRef, int $subjectRef)
    {
        ## check class existence in the Class_fee_schedule table.
        $ClassFeeScheduler = $this->entity_manager->getRepository(ClassFeeSchedule::class)->findBy(
            ['class' => $classRef, 'class_subject' => $subjectRef]
        );


        $response = 'escape__fromErro_replaceAllContent'.$this->environment->render(
                'fee/ClassFeeScheduler/classFeeSchedulerEditInterface.html.twig',
                [
                    'FeesList' => $ClassFeeScheduler,
                    'from_subject' => true,
                ]
            );

        return $response;
    }

    /**
     * Table:Class Session enrolment fee Schedule.
     * */

    ## save class scheduler for the student.
    ## all student have implemented default fees - but some special student has implemented special fees.
    public function _save_class_fee_schedule_for_special_student(ParameterBag $request)
    {
        $response = [];

        ## check validate response.
        if (!empty($validation_response = $this->validate__class_fee_scheduler($request))) {
            return $validation_response;
        }

        $Entity = new ClassSessionEnrolmentFeeSchedule();

        ## get the class.
        $Class = $this->enrollments->get__class_session_enrolment($request->get('c_r'));
        //$Class = $this->student_classes->get_the_class( $request->get( 'c_r' ) );
        if (!$Class instanceof ClassSessionEnrolment) {
            $response = $this->default_function->push_error($response, $Class);
        }

        ## set amount
        $Entity->setCsefsAmount((float)$request->get('am'));

        ## get the fee reference if not empty
        if (!empty($request->get('f_r'))) {
            $Fee = $this->student_fee_service->getFeeFromClassFeeTable($request->get('f_r'));
            if (!$Fee instanceof ClassFee) {
                $response = $this->default_function->push_error($response, $Fee);
            } else {
                ## save amount of fee
                $Entity->setCsefsAmount((float)$Fee->getAmount());
                ## save class fee
                $Entity->setFee($Fee);
            }
        }


        ## if error then send it back
        if (!empty($response)) {
            return $response;
        }

        $Entity->setCsefsTitle($request->get('tie'));
        $Entity->setClassSessionEnrolment($Class);
        $Entity->setCsefsGenerationDate($this->default_function->convert_datetimeStringIntoObject($request->get('gd')));
        $Entity->setCsefsDueDate($this->default_function->convert_datetimeStringIntoObject($request->get('dd')));
        $Entity->setCsefsLastDueDate($this->default_function->convert_datetimeStringIntoObject($request->get('ldd')));
        $Entity->setCsefsLastProcessDate($this->default_function->convert_datetimeStringIntoObject(''));
        $Entity->setProcess(1);

        try {
            $this->entity_manager->persist($Entity);
            $this->entity_manager->flush();
            $response = 'OK';
        } catch (\Exception $exception) {
            $response = $this->default_function->push_error($response, $exception->getMessage());
        }

        return $response;
    }

    ## get class session enrolment fee schedule.
    public function get_class_session_enrolment_fee_schedule(int $recordID)
    {
        $response = null;
        if (!empty($recordID)) {
            try {
                $response = $this->class_session_enrolment_fee_schedule_repository->find($recordID);
            } catch (\Exception $exception) {
                $response = $exception->getMessage();
            }
        }

        return $response;
    }

    ## delete fee schedule of special student.
    public function delete_class_session_enrolment_fee_schedule(int $recordID)
    {
        $response = 'Invalid reference';
        if (!empty($recordID)) {
            ## get the target record.
            $response = $this->get_class_session_enrolment_fee_schedule($recordID);
            if ($response instanceof ClassSessionEnrolmentFeeSchedule) {
                ## if record exits.
                $this->entity_manager->remove($response);
                $this->entity_manager->flush();
                $response = 'OK';
            } else {
                if (empty($response)) {
                    $response = 'No, such record found';
                }
            }
        }

        return $response;
    }

}
