<?php


namespace App\Service\EmailSettings;


use App\Entity\User;
use App\Service\AppSettings;
use Doctrine\ORM\Query\Parameter;
use PhpParser\Node\Param;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\HttpFoundation\ParameterBag;
use Symfony\Component\Routing\RouterInterface;
use Symfony\Component\Security\Csrf\TokenStorage\TokenStorageInterface;
use Twig\Environment;
use Twig\Error\LoaderError;
use Twig\Error\RuntimeError;
use Twig\Error\SyntaxError;

class Base extends Settings {

	/**
	 * @var \Swift_Mailer
	 */
	private $swift_Mailer;
	/**
	 * @var Environment
	 */
	private $environment;

	/**
	 * Base constructor.
	 *
	 * @param Environment           $environment
	 * @param \Swift_Mailer         $swift_Mailer
	 * @param AppSettings           $app_settings
	 * @param RouterInterface       $router
	 * @param TokenStorageInterface $token_storage
	 * @param ParameterBagInterface $parameter_bag
	 */
	public function __construct( Environment $environment, \Swift_Mailer $swift_Mailer, AppSettings $app_settings, RouterInterface $router, TokenStorageInterface $token_storage, ParameterBagInterface $parameter_bag ) {
		parent::__construct( $app_settings, $router, $token_storage, $parameter_bag );
		## set email url
		$this->setMailURL();
		$this->swift_Mailer = $swift_Mailer;
		$this->environment  = $environment;
	}


	/**
	 * @param ParameterBag $bag
	 * Validate Email Parameters before send
	 *
	 * @return string
	 */
	public function validateBeforeEmailSend( ParameterBag $bag ) {
		if ( empty( $bag->get( 'subject' ) ) || empty( $bag->get( 'message' ) || empty( $bag->get( 'receiver' ) ) ) ) {
			return 'Subject, Receiver or Message is missing';
		}
	}


	/**
	 * @param ParameterBag $bag
	 * Purpose: Send Email
	 *
	 * @return bool|int|string|null
	 */
	public function sendEmail( ParameterBag $bag ) {
		if ( ! empty( $validate = $this->validateBeforeEmailSend( $bag ) ) ) {
			return $validate;
		}

		## Make Over Email Message according to the template design.
		$bag->set( 'message', $this->makeUpEmailMessage( $bag ) );

		$response = null;
		if ( empty( $this->getMailVendor() ) ) {
			$response = $this->sendEmailUsingWebServer( $bag );
		} else {
			$response = $this->sendEmailUsingSwiftMailer( $bag );
		}

		return $response;
	}

	/**
	 * @param ParameterBag $bag
	 * Purpose: Send Email using swift mailer
	 *
	 * @return int
	 */
	public function sendEmailUsingSwiftMailer( ParameterBag $bag ) {
		## Message
		$message = ( new \Swift_Message( $bag->get( 'subject' ) ) )->setFrom( $this->getMessageSentFromEmail() )
		                                                           ->setTo( $bag->get( 'receiver' ) )
		                                                           ->setBody( $bag->get( 'message' ) );
		## Attachments.
		if ( is_array( $bag->get( 'attachmentsPath' ) ) ) {
			foreach ( $bag->get( 'attachmentsPath' ) as $key => $value ) {
				$message->attach( \Swift_Attachment::fromPath( $value ) );
			}
		}

		return $this->swift_Mailer->send( $message );
	}


	/**
	 * @param ParameterBag $bag
	 * Purpose: Send Email using webServer
	 *
	 * @return bool|string
	 */
	public function sendEmailUsingWebServer( ParameterBag $bag ) {

		$headers = "From: fekara@sms.com\r\n";
		$headers .= "Reply-To: no-reply@fekara.com\r\n";
		$headers .= "CC: sulaman@sulaman.pk\r\n";
		$headers .= "MIME-Version: 1.0\r\n";
		$headers .= "Content-Type: text/html; charset=UTF-8\r\n";

		try {
			$response = mail( $bag->get( 'receiver' ), $bag->get( 'subject' ), $bag->get( 'message' ), $headers );
		} catch ( \Exception $exception ) {
			$response = $exception->getMessage();
		}

		return $response;

	}


	/**
	 * @param ParameterBag $bag
	 * Purpose: Making Email Template.
	 *
	 * @return mixed|string
	 */
	public function makeUpEmailMessage( ParameterBag $bag ) {
		$emailBody = $bag->get( 'message' );

		## find variables in the email content & parse them into relevant data.
		preg_match_all( "#\{{(.*?)\}}#", $bag->get( 'message' ), $variableInTemplate );
		if ( $bag->get( 'user' ) instanceof User ) {
			$ar = [
				'{{ fname }}' => $bag->get( 'user' )->getFname(),
				'{{ lname }}' => $bag->get( 'user' )->getLName(),
			];

			$emailBody = strtr( $bag->get( 'message' ), $ar );
		}

		try {
			$emailBody = $this->environment->render( 'notification/emailNotifier.html.twig', [
				'content' => $emailBody,
			] );
		} catch ( LoaderError $e ) {
			$emailBody = $bag->get( 'message' );
		} catch ( RuntimeError $e ) {
			$emailBody = $bag->get( 'message' );
		} catch ( SyntaxError $e ) {
			$emailBody = $bag->get( 'message' );
		}

		return $emailBody;

	}
}