<?php

namespace App\Controller;

use App\Entity\AppSettings;
use App\Entity\EmailTemplateEvents;
use App\Entity\EmailTemplates;
use App\Entity\ShortEmailTemplates;
use App\Repository\AppSettingsRepository;
use App\Repository\EmailTemplateEventsRepository;
use App\Repository\EmailTemplatesRepository;
use App\Repository\ShortEmailTemplatesRepository;
use App\Service\DefaultFunction;
use Gedmo\Translatable\Document\Repository\TranslationRepository;
use phpDocumentor\Reflection\DocBlock\Serializer;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Serializer\Encoder\JsonEncoder;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Component\Validator\Constraints\Json;
use Gedmo\Translatable\Translatable;

/**
 * @IsGranted("ROLE_USER")
 * */
class  EmailTemplatesController extends AbstractController {

	/**
	 * @var EmailTemplatesRepository
	 */
	private $email_templates_repository;
	/**
	 * @var ShortEmailTemplatesRepository
	 */
	private $short_email_templates_repository;
	/**
	 * @var \App\Service\AppSettings
	 */
	private $app_settings;

	public function __construct( EmailTemplatesRepository $email_templates_repository, ShortEmailTemplatesRepository $short_email_templates_repository, \App\Service\AppSettings $app_settings ) {
		$this->email_templates_repository       = $email_templates_repository;
		$this->short_email_templates_repository = $short_email_templates_repository;
		$this->app_settings                     = $app_settings;
	}

	/**
	 * @Route("{_locale}/set/request-templates/", name="request_template")
	 * @param Request                  $request
	 * @param EmailTemplatesRepository $email_templates_repository
	 * @param DefaultFunction          $default_function
	 *
	 * @return JsonResponse|Response
	 */
	public function ajaxHandler( Request $request, EmailTemplatesRepository $email_templates_repository, DefaultFunction $default_function ) {

		$returnResponse = new JsonResponse( 'Invalid request', 404 );

		switch ( $request->request->get( 'requestFor' ) ) {
			case 'dropdown':

				$TemplateList = $email_templates_repository->findBy( [ 'assigned_to' => $request->request->get( 'assigned_to' ) ] );
				## find the translation of the first template
				if ( ! empty( $TemplateList[0] ) ) {
					$translationRepo = $this->before__fetchingListOfField( $TemplateList[0] );
				} else {
					$translationRepo = '';
				}

				## populate dropdown, with the email template list.
				$returnResponse = $this->render( 'email_templates/ajaxPatches/selectTemplateToEditDropdown.html.twig', [
					'emailTemplates'     => $TemplateList,
					'translationDetails' => $translationRepo,
				] );

				break;
			case 'email_template_details':
				## details of email template
				$template = $this->decide_where_to_get_template( $request->request->get( 'id' ), $request->request->get( 'template_type' ) );
				if ( ! empty( $template ) ) {
					$translationRepo = $this->getDoctrine()
					                        ->getManager()
					                        ->getRepository( 'Gedmo\\Translatable\\Entity\\Translation' );

					## fetching translation of given, template.
					$translationRepo = $translationRepo->findTranslations( $template );

					## if client requested for the single lang & we found that language in database then return
					if ( ! empty( $request->request->get( 'lang' ) ) && ! empty( $translationRepo[ $request->request->get( 'lang' ) ] ) ) {
						$translationRepo = $translationRepo[ $request->request->get( 'lang' ) ];
					}

					## if we found no translation against request then create an empty array to prevent errors.
					if ( empty( $translationRepo ) ) {
						$translationRepo['content'] = '';
						$translationRepo['subject'] = '';
					}

					$returnResponse = new JsonResponse( $translationRepo, 200 );
				}
				break;
			case 'saveTemplate':
				## save multi-lingual template
				$response = [];
				if ( $request->request->get( 'template_type' ) == 'detail_notification' && empty( $request->request->get( 'subject' ) ) ) {
					$response = $default_function->push_error( $response, 'Subject of notification is empty' );
				}

				if ( empty( $request->request->get( 'content' ) ) ) {
					$response = $default_function->push_error( $response, 'Content of notification is empty' );
				}

				## if we've no errors then proceed.
				if ( empty( $response ) ) {

					## get the template depending on the notification type.
					$template = $this->decide_where_to_get_template( $request->request->get( 'id' ), $request->request->get( 'template_type' ) );
					if ( ! empty( $template ) ) {
						try {
							## laod translate repository
							$GedmoTranslator = $this->getDoctrine()
							                        ->getManager()
							                        ->getRepository( 'Gedmo\\Translatable\\Entity\\Translation' );

							## save email subject if not exits.
							if ( ! empty( $request->request->get( 'subject' ) ) ) {
								## save subject of email template.
								$GedmoTranslator->translate( $template, 'subject', $request->request->get( 'lang' ), $request->request->get( 'subject' ) );
							}
							## save content of the template.
							$GedmoTranslator->translate( $template, 'content', $request->request->get( 'lang' ), $request->request->get( 'content' ) );

							## save data to database.
							$this->getDoctrine()->getManager()->persist( $template );
							$this->getDoctrine()->getManager()->flush();

							$returnResponse = new JsonResponse( 'OK', 200 );

						} catch ( \Exception $exception ) {
							$returnResponse = new JsonResponse( $exception->getMessage() );
						}
					}
				} else {
					$returnResponse = new JsonResponse( $response, 200 );
				}

				break;
			case 'updateEmailTemplateActiveInActiveStatus':
				## update template status, active or in-active.
				$template = $this->decide_where_to_get_template( $request->request->get( 'id' ), $request->request->get( 'notificationType', 'detail_notification' ) );
				if ( ! empty( $template ) ) {
					## if email template is active then system will de-active or vice versa.
					$template->setIsActive( ! $template->getIsActive() );
					## save data to the database.
					$this->getDoctrine()->getManager()->persist( $template );
					$this->getDoctrine()->getManager()->flush();
					$returnResponse = new JsonResponse( 'OK', 200 );
				}
				break;
		};

		return $returnResponse;
	}

	/*
	 * We are saving two type of templates.
	 * 1 is email template.
	 * 2 is short email template.
	 * so, this method will help us, to get the template from the required database table.
	 * */
	private function decide_where_to_get_template( ?int $templateID, ?string $messageType ) {
		if ( $messageType == 'detail_notification' ) {
			## load template from email_templates table.
			$template = $this->email_templates_repository->find( $templateID );
			if ( ! $template instanceof EmailTemplates ) {
				$template = null;
			}
		} else {
			## load template from the short message template.
			$template = $this->short_email_templates_repository->findOneBy( [ 'email_template_refer' => $templateID ] );
			if ( ! $template instanceof ShortEmailTemplates ) {
				$template = null;
			}
		}

		return $template;
	}

	/**
	 *
	 * @Route("{_locale}/settings/email-templates/{query}", name="email_templates", defaults={"query":null})
	 * @param EmailTemplatesRepository $email_templates_repository
	 * @param \App\Service\AppSettings $app_settings
	 * @param int                      $query
	 *
	 * @return Response
	 */
	public function index( EmailTemplatesRepository $email_templates_repository, $query = 0 ) {
		if ( empty( $query ) ) {
			## if client request request for nothing so, load first tab template which is common template.
			$criteria = [ 'assigned_to' => 'ct' ];
		} else {
			## if client request for the single template.
			$criteria = [ 'id' => $query ];
		}

		## find the template according to the criteria
		$EmailTemplates = $email_templates_repository->findBy( $criteria );

		## find the translation of the first template.
		if ( ! empty( $EmailTemplates[0] ) ) {
			$translationRepo = $this->before__fetchingListOfField( $EmailTemplates[0] );
		} else {
			## to prevent undefined error.
			$translationRepo = '';
		}

		return $this->render( 'email_templates/index.html.twig', [
			'emailTemplates'     => $EmailTemplates,
			'translationDetails' => $translationRepo,
		] );
	}


	/*
	 *
	 * */
	public function before__fetchingListOfField( $template ) {

		## if we found the template in the database then get the first template translations from the translation table.
		if ( isset( $template ) ) {
			## get the Gedmo library
			$translationRepo = $this->getDoctrine()
			                        ->getManager()
			                        ->getRepository( 'Gedmo\Translatable\Entity\Translation' );

			## fetching translation of given, template.
			$translationRepo = $translationRepo->findTranslations( $template );
			## get the translation according to the application default locale,
			## if default locale is not found then system will get the english template.
			return $translationRepo[ $this->app_settings->getAppSettingsFromSessions( 'default_language' ) ];
		}
	}


}
