<?php

namespace App\Repository;

use App\Entity\ClassSubject;
use App\Entity\ExamsClasses;
use App\Entity\ExamsClassesStudent;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\NoResultException;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr\Join;
use Doctrine\ORM\Query\QueryException;
use Nzo\UrlEncryptorBundle\NzoUrlEncryptorBundle;
use Nzo\UrlEncryptorBundle\UrlEncryptor\UrlEncryptor;
use Symfony\Bridge\Doctrine\RegistryInterface;
use Symfony\Component\HttpFoundation\ParameterBag;

/**
 * @method ExamsClasses|null find($id, $lockMode = null, $lockVersion = null)
 * @method ExamsClasses|null findOneBy(array $criteria, array $orderBy = null)
 * @method ExamsClasses[]    findAll()
 * @method ExamsClasses[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 */
class ExamsClassesRepository extends ServiceEntityRepository
{

    /**
     * @var RegistryInterface
     */
    private $registry;
    /**
     * @var UrlEncryptor
     */
    private $url_encryptor;

    public function __construct(RegistryInterface $registry, UrlEncryptor $url_encryptor)
    {
        parent::__construct($registry, ExamsClasses::class);
        $this->registry = $registry;
        $this->url_encryptor = $url_encryptor;
    }


    ## Distinct Locations from Exam Class table.
    public function distinctExamClassLocation()
    {
        return $this->createQueryBuilder('examsClasses')
            ->select('examsClasses.exam_location')
            ->distinct(true)
            ->getQuery()
            ->getResult();
    }

    /**
     * Class exam duplication check.
     * duplication check on base of  1, Exam 2, class 3, Subject , 4 Date
     *
     * @param int $examName
     * @param int $classRef
     * @param int $classSubject
     * @param string $date
     * @param int $findNotByThisID
     *
     * @return mixed
     */
    public function saveClassExamDuplicationCheck(
        int $examName,
        int $classRef,
        int $classSubject,
        string $date,
        int $findNotByThisID = null
    ) {
        $queryBuilder = $this->createQueryBuilder('examsClasses')
            ->where('examsClasses.exam = :examName')
            ->setParameter('examName', $examName)
            ->andWhere('examsClasses.class = :Class')
            ->setParameter('Class', $classRef)
            ->andWhere('examsClasses.ClassSubject = :classSubject')
            ->setParameter('classSubject', $classSubject)
            ->andWhere('examsClasses.ec_date LIKE :date')
            ->setParameter('date', date('Y-m-d', strtotime($date)).' %');

        if (!empty($findNotByThisID)) {
            $queryBuilder->andWhere('examsClasses.id <> :id')->setParameter('id', $findNotByThisID);
        }

        return $queryBuilder->getQuery()->getResult();

    }


    ## Advance check
    public function examClassAdvanceSearch(ParameterBag $bag)
    {

        $queryBuilder = $this->createQueryBuilder('examsClasses');
        $queryBuilder->leftJoin('examsClasses.class', 'class')
            ->addSelect('class')
            ->leftJoin('class.session', 'session')
            ->addSelect('session');
        $queryBuilder->leftJoin('examsClasses.examiners', 'examiners')->addSelect('examiners');


        ## multiple classes search
        if (!empty($bag->get('classes'))) {
            $queryBuilder->add(
                'where',
                $queryBuilder->expr()
                    ->in('examsClasses.class', $bag->get('classes'))
            );
        }

        ## search session
        if (!empty($bag->get('se'))) {
            $queryBuilder->andWhere('session = :sessionRef')->setParameter('sessionRef', $bag->get('se'));
        }

        ## class search
        if (!empty($bag->get('cl'))) {
            $queryBuilder->andWhere('examsClasses.class = :classRef')->setParameter('classRef', $bag->get('cl'));
        }


        ## subject search
        if (!empty($bag->get('su'))) {
            $queryBuilder->andWhere('examsClasses.ClassSubject = :classSubjectRef')
                ->setParameter('classSubjectRef', $bag->get('su'));
        }

        ## exam date search
        if (!empty($bag->get('eD'))) {
            $queryBuilder->andWhere('examsClasses.ec_date LIKE :examDate')
                ->setParameter('examDate', date('Y-m-d', strtotime($bag->get('eD'))).'%');
        }

        ## location search
        if (!empty($bag->get('loc'))) {
            $queryBuilder->andWhere('examsClasses.exam_location = :examLocation')
                ->setParameter('examLocation', $bag->get('loc'));
        }

        ## Examiner search
        if (!empty($bag->get('ex'))) {
            $queryBuilder->andWhere('examiners.user = :examiner')->setParameter('examiner', $bag->get('ex'));
        }

        ## search by exam
        if (!empty($bag->get('exam'))) {
            $queryBuilder->andWhere('examsClasses.exam = :examRef')->setParameter('examRef', $bag->get('exam'));
        }

        $queryBuilder->orderBy('examsClasses.id', 'DESC');

        ## get result.
        return $queryBuilder->getQuery()->getResult();
    }


    ## mark exams results
    public function markExamsResults($examClass)
    {
        try {
            return $this->createQueryBuilder('examsClasses')
                ->leftJoin('examsClasses.class', 'class')
                ->addSelect('class')
                ->leftJoin('class.classSessionEnrolments', 'classSessionEnrolments')
                ->addSelect('classSessionEnrolments')
                ->leftJoin('classSessionEnrolments.user_id', 'userId')
                ->addSelect('userId')
                ->leftJoin(
                    'examsClasses.examsClassesStudents',
                    'examsClassesStudents',
                    Join::WITH,
                    'userId.id = examsClassesStudents.exam_did_by OR examsClassesStudents.id is null'
                )
                ->addSelect('examsClassesStudents')
                ->andwhere('examsClasses.id = :examClassRef')
                ->setParameter('examClassRef', $examClass)
                ->getQuery()
                ->getOneOrNullResult();
        } catch (NonUniqueResultException $e) {
            return $e->getMessage();
        }
    }


    ## get the list of exams Class by the Class Ref
    public function getTheScheduledExamsListByRefOfClass(
        $ClassRef = null,
        $examRef = null,
        $arrayKeyAsClassSubject = true
    ) {

        try {
            $results = $this->createQueryBuilder('examsClass')
                ->select('examsClass')
                ->leftJoin('examsClass.ClassSubject', 'class_subject')
                ->addSelect('class_subject')
                ->andwhere('examsClass.class = :classRef')
                ->andWhere('examsClass.exam = :examRef')
                ->setParameter('examRef', $examRef)
                ->setParameter('classRef', $ClassRef)
                ->getQuery()
                ->getResult();

            $ar = [];
            if ($arrayKeyAsClassSubject && !empty($results)) {
                $ar = [];
                /** @var ExamsClasses $value */
                foreach ($results as $value) {
                    $ar[$value->getClassSubject()->getID()] = $value;
                }
            }

            return $ar;
        } catch (QueryException $e) {
            return $e->getMessage();
        }

    }


}
