<?php

namespace App\Controller;

use App\Form\UserLoginFormType;
use App\Form\UserRegistrationType;
use App\Repository\CustomFieldsRepository;
use App\Repository\UserRepository;
use App\Security\ApiTokenGenerator;
use App\Service\AppSettings;
use App\Service\CustomTokenManager;
use App\Service\DefaultFunction;
use App\Service\UserRegistration;
use App\Service\UserService;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Security\Http\Authentication\AuthenticationUtils;

class SecurityController extends AbstractController
{

    /**
     * @var UserRepository
     */
    private $user_repository;
    /**
     * @var CustomFieldsRepository
     */
    private $custom_fields_repository;

    /**
     * @var UserRegistrationType
     * */
    protected $userRegistrationForm;

    const fieldsType = [
        's' => 1,
        't' => 2,
        'g' => 3,
        'a' => 4,
    ];
    /**
     * @var AppSettings
     */
    private $app_settings;

    /**
     * SecurityController constructor.
     * @param UserRepository $user_repository
     * @param CustomFieldsRepository $custom_fields_repository
     * @param AppSettings $app_settings
     */
    public function __construct(
        UserRepository $user_repository,
        CustomFieldsRepository $custom_fields_repository,
        AppSettings $app_settings
    ) {
        $this->user_repository = $user_repository;
        $this->custom_fields_repository = $custom_fields_repository;
        $this->app_settings = $app_settings;

    }

    /**
     * @Route("{_locale}/", name="user_login_locale")
     * @Route("/", name="user_login")
     * @param AuthenticationUtils $authentication_utils
     *
     * @return string|RedirectResponse
     */
    public function index(AuthenticationUtils $authentication_utils)
    {
        ## show Sign In page is user is logged in.
        if ($response = $this->do_not_show_untill_logout()) {
            return $response;
        }

        ## make login form
        $loginForm = $this->createForm(UserLoginFormType::class);

        return $this->render(
            'security/index.html.twig',
            [
                'controller_name' => 'SecurityController',
                'loginForm' => $loginForm->createView(),
                'error' => $authentication_utils->getLastAuthenticationError(),
                'authByValue' => $authentication_utils->getLastUsername(),
            ]
        );
    }

    /**
     * @Route("/security/student-registration", name="student_regisration")
     * @Route("{_locale}/security/student-registration", name="student_regisration_locale")
     */
    public function studentRegistration()
    {
        if ($this->app_settings->getDataFromAppSettings('studentRegistrationOnline', true) == 'no') {
            return $this->redirectToRoute('user_login');
        }

        ## show Sign In page is user is logged in.
        if ($response = $this->do_not_show_untill_logout()) {
            return $response;
        }

        ## getting the registration field of students
        $studentCustomFields = $this->custom_fields_repository->getUserRegistrationField('1');

        return $this->render(
            'security/registration.html.twig',
            [
                'category' => 's',
                //'registrationForm' => $userRegistrationForm->createView(),
                'fieldLIst' => $studentCustomFields,
                'registrationType' => 'Student',
            ]
        );

    }

    /**
     * @Route("/security/admin-registration", name="admin_regisration")
     * @Route("{_locale}/security/admin-registration", name="admin_regisration_locale")
     */
    public function administrationRegistration()
    {
        if ($this->app_settings->getDataFromAppSettings('teacherRegistrationOnline', true) == 'no') {
            return $this->redirectToRoute('user_login');
        }

        ## show Sign In page is user is logged in.
        if ($response = $this->do_not_show_untill_logout()) {
            return $response;
        }


        ## getting the registration field for the administrator
        $administratorCustomFields = $this->custom_fields_repository->getUserRegistrationField('4');

        return $this->render(
            'security/registration.html.twig',
            [
                'controller_name' => 'SecurityController',
                'category' => 'a',
                'fieldLIst' => $administratorCustomFields,
                'registrationType' => 'Admin',
            ]
        );
    }

    /**
     * @Route("/security/teacher-registration", name="teacher_registration")
     * @Route("{_locale}/security/teacher-registration", name="teacher_registration_locale")
     */
    public function teacherRegistration()
    {

        if ($this->app_settings->getDataFromAppSettings('teacherRegistrationOnline', true) == 'no') {
            return $this->redirectToRoute('user_login');
        }

        ## show Sign In page is user is logged in.
        if ($response = $this->do_not_show_untill_logout()) {
            return $response;
        }

        ## getting the registration field for the teachers
        $teacherCustomFields = $this->custom_fields_repository->getUserRegistrationField('2');

        return $this->render(
            'security/registration.html.twig',
            [
                'controller_name' => 'SecurityController',
                'category' => 't',
                'fieldLIst' => $teacherCustomFields,
                'registrationType' => 'Teacher',
            ]
        );
    }

    /**
     * @Route("/security/save-registration-form", name="save_user_registration_form")
     * @Route("{_locale}/security/save-registration-form", name="save_user_registration_form_locale")
     * @param Request $request
     * @param UserRegistration $user_registration
     * @param DefaultFunction $defaultFunction
     *
     * @return JsonResponse
     */
    public function saveUserRegistrationForm(
        Request $request,
        UserRegistration $user_registration,
        DefaultFunction $defaultFunction
    ) {
        try {
            $response = $user_registration->registerUser($request);
            $response = $response.$defaultFunction->redirectToHomePage();
        } catch (\Exception $exception) {
            // thrown an error.
            $response = $exception->getMessage();
        }

        ## if user register successfully - the redirect to the homePage.
        return new JsonResponse($response, 200);
    }


    /**
     * @Route("{_locale}/security/logout", name="security_logout_locale")
     * @Route("/security/logout", name="security_logout")
     */
    public function logout()
    {
    }

    /**
     * @Route("{_locale}/guardian/login/", name="guardian_login_locale")
     * @Route("/guardian/login/", name="guardian_login")
     * @param AuthenticationUtils $authenticationUtils
     *
     * @return Response
     */
    public function guardian_login(AuthenticationUtils $authenticationUtils): Response
    {

        // get the login error if there is one
        $error = $authenticationUtils->getLastAuthenticationError();
        // last username entered by the user
        $lastUsername = $authenticationUtils->getLastUsername();

        return $this->render(
            'security/guardianLogin.html.twig',
            [
                'authByValue' => $lastUsername,
                'error' => $error,
            ]
        );
    }

    ## this method is used to redirect the user to dashboard page if user is logged in.
    private function do_not_show_untill_logout()
    {
        $response = '';
        ## if user is already logged in then redirect to the home page.
        If (!empty($this->getUser()) && in_array('ROLE_USER', $this->getUser()->getRoles())) {
            $defaultLocale = $this->app_settings->getAppSettingsFromSessions('default_language');

            $response = $this->redirectToRoute('HomePage', ['_locale' => $defaultLocale]);
        }

        return $response;
    }

    /**
     * @Route("{_locale}/security/forgot-password", name="forgot__password_locale")
     * @Route("/security/forgot-password", name="forgot__password")
     */
    ## forgot password
    public function request_for_forgot_password_interface()
    {
        return $this->render('security/forgot__password.html.twig');
    }

    ## execute forgot password request

    /**
     * @Route("/security/save-reset-form", name="save_user_reset_form")
     * @Route("{_locale}/security/save-reset-form", name="save_user_reset_form_locale")
     * @param Request $request
     * @param UserService $user_service
     *
     * @return RedirectResponse
     */
    public function saveForgotPasswordForm(Request $request, UserService $user_service)
    {
        try {
            $response = $user_service->request___forgot_password($request->request);
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }

        if ($response <> 'OK') {
            $this->addFlash('forgotPasswordFlash', json_encode($response));
        }

        return $this->redirectToRoute('forgot__password');
        ## return response in json
//        return new JsonResponse($response, 200);
    }

    ## asking new password when click on the reset password.

    /**
     * @Route("/security/update-password/{token__string}", name="security__update_my_pass_by_resetPass")
     * @Route("{_locale}/security/update-password/{token__string}", name="security__update_my_pass_by_resetPass_locale")
     * @param CustomTokenManager $custom_token_manager
     * @param null $token__string
     *
     * @return Response
     * @throws \Exception
     */
    public function updatePassByForgotPassword(CustomTokenManager $custom_token_manager, $token__string = null)
    {
        if (!empty($token__string)) {
            ## if token is not empty

            ## validate the token
            $response = $custom_token_manager->is__token__valid($token__string, true);

            ## if toke is valid
            if (isset($response['response']) && $response['response']) {
                return $this->render('security/update__password.html.twig');
            }
        }

    }

    ## update password

    /**
     * @Route("/security/update-password-remote-request", name="update_password_request")
     * @Route("{_locale}/security/update-password-remote-request", name="update_password_request_locale")
     * @param Request $request
     * @param UserService $user_service
     *
     * @return JsonResponse
     */
    public function udpatePasswordRequest__(Request $request, UserService $user_service)
    {
        $response = null;
        try {
            $response = $user_service->updateUserPassword__pre__action($request->request);
        } catch (\Exception $exception) {
            $response = $exception->getMessage();
        }


        ## return response in json
        return new JsonResponse($response, 200);
    }

    /**
     * @Route("api/security/api-login", name="api_login_route")
     * @param Request $request
     * @param ApiTokenGenerator $apiTokenGenerator
     * @return void
     */
    public function ApiAuthenticator(Request $request, ApiTokenGenerator $apiTokenGenerator)
    {
        $apiTokenGenerator->generateToken($request);
        dd($request->request);
    }
}
